// value with derivatives

// The "Value" is supposed to be a function of paramters t_1,...,t_n
// This structure stores the value of the "Value" and its n derivatives.

#ifndef VWD_LIB
#define VWD_LIB

//#define WANT_ERF                         // activate if you want erf
                                         // you'll need erf in your library

#include "newmat.h"                      // newmat functions
#include "str.h"                         // string class


// **************** the parameter set manipulation classes *******************

// ParameterSet PS();                   start a new parameter set
// int i = PS.AddParameter("name");     add a new parameter called "name"
//                                      check it for uniqueness
//                                      return its index
// int i = PS.LocateParameter("name");  find index number (1,...,n) of "name"
//                                      return 0 if not present
// String name = PS(i);                 return name with index i
// int n = PS.Size();                   return number of parameters
// bool f = PS.Frozen();                return true if number of parameters
//                                      is "frozen" - not working yet
// bool e = PS1 == PS2;                 return true if pointing to same
//                                      ParameterSetClass
// bool e = PS1 != PS2;                 return true if not pointing to same
//                                      ParameterSetClass
// AssertEqual(PS1, PS2);               throw an exception if not pointing
//                                      to same underlying class


// a list of strings

class NameSeq
{
public:
   NameSeq* Next;
   String Name;
   NameSeq(const String& name) : Next(0), Name(name) {}
};

// the real data for the ParameterSet

class ParameterSetClass
{
   bool Frozen;
   long RefCount;
   int n;
   NameSeq* Next;
   int last_i;                          // details of last name searched for
   NameSeq* last_seq;
   ParameterSetClass()
      : Frozen(false), RefCount(0), n(0), Next(0), last_i(0) {}
   ~ParameterSetClass() { CleanUp(); }
   void CleanUp();
   void IncrRef() { RefCount++; }
   void DecrRef();
   int AddParameter(const String& name);
   int LocateParameter(const String& name) const;
   String& operator()(int i);
   friend class ParameterSet;
};

// the parameter set as seen by the user

class ParameterSet : public Janitor
{
   ParameterSetClass* PSC;
   void CleanUp() { PSC->DecrRef(); }
public:
   ParameterSet();
   ParameterSet(const ParameterSet& ps) : PSC(ps.PSC) { PSC->IncrRef(); }
   ~ParameterSet() { PSC->DecrRef(); }
   void operator=(const ParameterSet& ps);
   int AddParameter(const String& name) { return PSC->AddParameter(name); }
   int LocateParameter(const String& name) const
      { return PSC->LocateParameter(name); }
   String& operator()(int i) const { return PSC->operator()(i); }
   int Size() const { return PSC->n; }
   bool Frozen() const { return PSC->Frozen; }
   friend bool operator==(const ParameterSet& ps1, const ParameterSet& ps2)
      { return ps1.PSC == ps2.PSC; }
   friend bool operator!=(const ParameterSet& ps1, const ParameterSet& ps2)
      { return ps1.PSC != ps2.PSC; }
   friend void AssertEqual(const ParameterSet& ps1, const ParameterSet& ps2);
};

// ********************** Value with Derviatives classes *********************

class RVWD;

class VWD
{
   Real Value;
   RowVector Derivatives;
   ParameterSet PS;
   static ParameterSet PS_null;                   // used by default constructor
public:
   VWD();
   explicit VWD(Real v);
   VWD(const ParameterSet& ps, Real v);
   VWD(const ParameterSet& ps, Real v, const RowVector& d);
   VWD(const ParameterSet& ps, Real v, int k);
   VWD(const ParameterSet& ps, Real v, const String& name);
   VWD(const VWD&);
   VWD(const RVWD&);
   ~VWD() {}
   Real GetValue() const { return Value; }
   ReturnMatrix GetDerivatives() const { return Derivatives; }
   Real GetDerivative(int i) const { return Derivatives(i); }
   ParameterSet GetParameterSet() { return PS; }
   void operator=(const VWD&);
   void operator+=(const VWD&);
   void operator-=(const VWD&);
   void operator*=(const VWD&);
   void operator/=(const VWD&);
   void operator+=(Real);
   void operator-=(Real);
   void operator*=(Real);
   void operator/=(Real);
   VWD operator-() const;
   friend void purge(VWD&) {}
   friend VWD operator+(const VWD&, const VWD&);
   friend VWD operator-(const VWD&, const VWD&);
   friend VWD operator*(const VWD&, const VWD&);
   friend VWD operator/(const VWD&, const VWD&);
   friend VWD pow(const VWD&, const VWD&);
   friend VWD operator+(Real, const VWD&);
   friend VWD operator-(Real, const VWD&);
   friend VWD operator*(Real, const VWD&);
   friend VWD operator/(Real, const VWD&);
   friend VWD pow(Real, const VWD&);
   friend VWD operator+(const VWD&, Real);
   friend VWD operator-(const VWD&, Real);
   friend VWD operator*(const VWD&, Real);
   friend VWD operator/(const VWD&, Real);
   friend VWD pow(const VWD&, Real);
   friend VWD pow(const VWD&, int);
   friend VWD exp(const VWD&);
   friend VWD log(const VWD&);
   friend VWD sin(const VWD&);
   friend VWD cos(const VWD&);
   friend VWD tan(const VWD&);
#ifdef WANT_ERF
   friend VWD erf(const VWD&);
#endif
};

// prototype of VWD which is a function of a Real

class VWDOfReal
{
protected:
   Real x;                             // Current x value
   bool xSet;                          // true if a value assigned to x

public:
   virtual void Set(Real X) { x = X; xSet = true; }
                                       // set x
   VWDOfReal() : xSet(false) {}
   virtual VWD operator()() = 0;
                                       // function value at current x
                                       // set current x
   VWD operator()(Real X) { Set(X); return operator()(); }
                                       // set x, return value
};

// Gaussian numerical integration
VWD GaussianIntegration32(VWDOfReal& function, Real Lower, Real Upper);

// ***** Value with Derviatives classes - first and second derivatives *****

class VWD2
{
   Real Value;
   RowVector Derivatives;
   Matrix SecondDerivatives;
   ParameterSet PS;
   static ParameterSet PS_null;                   // used by default constructor
public:
   VWD2();
   explicit VWD2(Real v);
   VWD2(const ParameterSet& ps, Real v);
   VWD2(const ParameterSet& ps, Real v, const RowVector& d, const Matrix& d2);
   VWD2(const ParameterSet& ps, Real v, int k);
   VWD2(const ParameterSet& ps, Real v, const String& name);
   VWD2(const VWD2&);
   ~VWD2() {}
   Real GetValue() const { return Value; }
   ReturnMatrix GetDerivatives() const { return Derivatives; }
   ReturnMatrix GetSecondDerivatives() const { return SecondDerivatives; }
   Real GetDerivative(int i) const { return Derivatives(i); }
   Real GetSecondDerivative(int i, int j) const
      { return SecondDerivatives(i, j); }
   ParameterSet GetParameterSet() { return PS; }
   void operator=(const VWD2&);
   void operator+=(const VWD2&);
   void operator-=(const VWD2&);
   void operator*=(const VWD2&);
   void operator/=(const VWD2&);
   void operator+=(Real);
   void operator-=(Real);
   void operator*=(Real);
   void operator/=(Real);
   VWD2 operator-() const;
   friend void purge(VWD2&) {}
   friend VWD2 operator+(const VWD2&, const VWD2&);
   friend VWD2 operator-(const VWD2&, const VWD2&);
   friend VWD2 operator*(const VWD2&, const VWD2&);
   friend VWD2 operator/(const VWD2&, const VWD2&);
   friend VWD2 pow(const VWD2&, const VWD2&);
   friend VWD2 operator+(Real, const VWD2&);
   friend VWD2 operator-(Real, const VWD2&);
   friend VWD2 operator*(Real, const VWD2&);
   friend VWD2 operator/(Real, const VWD2&);
   friend VWD2 pow(Real, const VWD2&);
   friend VWD2 operator+(const VWD2&, Real);
   friend VWD2 operator-(const VWD2&, Real);
   friend VWD2 operator*(const VWD2&, Real);
   friend VWD2 operator/(const VWD2&, Real);
   friend VWD2 pow(const VWD2&, Real);
   friend VWD2 pow(const VWD2&, int);
   friend VWD2 exp(const VWD2&);
   friend VWD2 log(const VWD2&);
   friend VWD2 sin(const VWD2&);
   friend VWD2 cos(const VWD2&);
   friend VWD2 tan(const VWD2&);
#ifdef WANT_ERF
   friend VWD2 erf(const VWD2&);
#endif
};

// prototype of VWD which is a function of a Real

class VWD2OfReal
{
protected:
   Real x;                             // Current x value
   bool xSet;                          // true if a value assigned to x

public:
   virtual void Set(Real X) { x = X; xSet = true; }
                                       // set x
   VWD2OfReal() : xSet(false) {}
   virtual VWD2 operator()() = 0;
                                       // function value at current x
                                       // set current x
   VWD2 operator()(Real X) { Set(X); return operator()(); }
                                       // set x, return value
};

// Gaussian numerical integration
VWD2 GaussianIntegration32(VWD2OfReal& function, Real Lower, Real Upper);

#endif

// body file: vwd.cpp


