// value with derivatives

// The "Value" is supposed to be a function of parameters t_1,...,t_n
// This structure stores the value of the "Value" and its n derivatives.


#define WANT_STREAM
#define WANT_MATH

#include "vwd.h"

#ifdef DO_REPORT
#define REPORT { static ExeCounter ExeCount(__LINE__,11); ++ExeCount; }
#else
#define REPORT {}
#endif



void ParameterSetClass::CleanUp()
{
   REPORT
   NameSeq* ns1;
   for (NameSeq* ns = Next; ns; ns = ns1) { ns1 = ns->Next; delete ns; }
   Frozen = false; n = 0; Next = 0; last_i = 0;
}

void ParameterSetClass::DecrRef()
{
   REPORT
   if (--RefCount < 0) { REPORT  delete this; }
}

ParameterSet::ParameterSet()
{
   REPORT
   PSC = new ParameterSetClass();
   if (!PSC) Throw(Bad_alloc());
}

void ParameterSet::operator=(const ParameterSet& ps)
{
   REPORT
   ParameterSetClass* PSC1 = PSC; PSC = ps.PSC;
   PSC->IncrRef(); PSC1->DecrRef();
}

int ParameterSetClass::AddParameter(const String& name)
{
   REPORT
   NameSeq* New = new NameSeq(name);          // element to be added to list
   if (!New) Throw(Bad_alloc());              // no space
   NameSeq* ns = Next;
   if (!ns) { REPORT Next = New; }            // no names on list
   else
   {
      REPORT
      for (NameSeq* ns1 = ns->Next; ns1; ns1 = ns1->Next)
      {
         if (ns->Name==name)
            Throw(Logic_error("Duplicate name in parameter set\n"));
         ns = ns1;
      }
      ns->Next = New;
   }
   return ++n;
}

int ParameterSetClass::LocateParameter(const String& name) const
{
   REPORT
   int i = 0;
   for (NameSeq* ns = Next; ns; ns = ns->Next)
      { i++; if (ns->Name==name) return i; }
   return 0;
}

String& ParameterSetClass::operator()(int i)
{
   // this rather complicated procedure is so if we call for names
   // in increasing orders of i we don't have to keep scanning through list
   REPORT
   if (i <=0 || i > n) Throw(Logic_error("Illegal parameter index\n"));
   NameSeq* ns;
   if (i < last_i || last_i == 0) { REPORT  ns = Next; last_i = 1; }
   else { REPORT  ns = last_seq; }
   while (last_i < i) { REPORT  ns = ns->Next; ++last_i; }
   last_seq = ns;
   return ns->Name;
}

void AssertEqual(const ParameterSet& ps1, const ParameterSet& ps2)
{
   REPORT
   if (ps1!=ps2) Throw(Logic_error("Different parameter sets\n"));
}


inline Real square(Real x) { return x * x; }

ParameterSet VWD::PS_null;                           // static variable

VWD::VWD()
   : Value(0), Derivatives(PS_null.Size()), PS(PS_null)
   { REPORT Derivatives = 0; }

VWD::VWD(Real v)
   : Value(v), Derivatives(PS_null.Size()), PS(PS_null)
   { REPORT Derivatives = 0; }

VWD::VWD(const ParameterSet& ps, Real v)
   : Value(v), Derivatives(ps.Size()), PS(ps)
   { REPORT Derivatives = 0; }

VWD::VWD(const ParameterSet& ps, Real v, const RowVector& d)
   : Value(v), Derivatives(d), PS(ps)
{
   REPORT
   if (PS.Size() != d.Ncols()) Throw(Logic_error("Incompatible dimensions\n"));
}

VWD::VWD(const ParameterSet& ps, Real v, int k)
   : Value(v), Derivatives(ps.Size()), PS(ps)
   { REPORT  Derivatives = 0; Derivatives(k) = 1; }

VWD::VWD(const ParameterSet& ps, Real v, const String& name)
   : Value(v), Derivatives(ps.Size()), PS(ps)
{
   REPORT
   Derivatives = 0;
   int k = PS.LocateParameter(name);
   if (!k) Throw(Logic_error("Name not recognised\n"));
   Derivatives(k) = 1;
}

VWD::VWD(const VWD& vwd)
   : Value(vwd.Value), Derivatives(vwd.Derivatives), PS(vwd.PS) { REPORT }

void VWD::operator=(const VWD& vwd)
   { REPORT  PS = vwd.PS; Value = vwd.Value; Derivatives = vwd.Derivatives; }

void VWD::operator+=(const VWD& vwd)
{
   REPORT
   if (PS == PS_null)
      { REPORT PS = vwd.PS; Derivatives.ReSize(PS.Size()); Derivatives = 0.0; }
   AssertEqual(PS, vwd.PS); Value += vwd.Value; Derivatives += vwd.Derivatives;
}

void VWD::operator-=(const VWD& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS); Value -= vwd.Value; Derivatives -= vwd.Derivatives;
}

void VWD::operator*=(const VWD& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Derivatives = vwd.Value * Derivatives + Value * vwd.Derivatives;
   Value *= vwd.Value;
}

void VWD::operator/=(const VWD& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS); Value /= vwd.Value;
   Derivatives = (Derivatives - Value  * vwd.Derivatives) / vwd.Value;
}

void VWD::operator+=(Real r) { REPORT  Value += r; }

void VWD::operator-=(Real r) { REPORT  Value -= r; }

void VWD::operator*=(Real r) { REPORT  Value *= r; Derivatives *= r; }

void VWD::operator/=(Real r) { REPORT  Value /= r; Derivatives /= r; }

VWD VWD::operator-() const { REPORT  return VWD(PS, -Value, -Derivatives); }

VWD operator+(const VWD& vwd1, const VWD& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return VWD(vwd1.PS, vwd1.Value + vwd2.Value,
      vwd1.Derivatives + vwd2.Derivatives);
}

VWD operator-(const VWD& vwd1, const VWD& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return VWD(vwd1.PS, vwd1.Value - vwd2.Value,
      vwd1.Derivatives - vwd2.Derivatives);
}

VWD operator*(const VWD& vwd1, const VWD& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return VWD(vwd1.PS, vwd1.Value * vwd2.Value,
      vwd2.Value * vwd1.Derivatives + vwd1.Value * vwd2.Derivatives);
}

VWD operator/(const VWD& vwd1, const VWD& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Real v = vwd1.Value / vwd2.Value;
   return VWD( vwd1.PS, v,
      (vwd1.Derivatives - v * vwd2.Derivatives) / vwd2.Value );
}

VWD pow(const VWD& vwd1, const VWD& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Real v = pow(vwd1.Value, vwd2.Value);
   return VWD( vwd1.PS, v,
      v * ((vwd2.Value / vwd1.Value) * vwd1.Derivatives
         + log(vwd1.Value) * vwd2.Derivatives) );
}

VWD operator+(Real r, const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, r + vwd.Value, vwd.Derivatives);
}

VWD operator-(Real r, const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, r - vwd.Value, - vwd.Derivatives);
}

VWD operator*(Real r, const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, r * vwd.Value, r * vwd.Derivatives);
}

VWD operator/(Real r, const VWD& vwd)
{
   REPORT
   Real v = r / vwd.Value;
   return VWD(vwd.PS, v, (- v / vwd.Value) * vwd.Derivatives);
}

VWD pow(Real r, const VWD& vwd)
{
   REPORT
   Real v = pow(r, vwd.Value);
   return VWD(vwd.PS, v, (v * log(r)) * vwd.Derivatives);
}

VWD operator+(const VWD& vwd, Real r)
{
   REPORT
   return VWD(vwd.PS, vwd.Value + r, vwd.Derivatives);
}

VWD operator-(const VWD& vwd, Real r)
{
   REPORT
   return VWD(vwd.PS, vwd.Value - r, vwd.Derivatives);
}

VWD operator*(const VWD& vwd, Real r)
{
   REPORT
   return VWD(vwd.PS, vwd.Value * r, vwd.Derivatives * r);
}

VWD operator/(const VWD& vwd, Real r)
{
   REPORT
   return VWD(vwd.PS, vwd.Value / r, vwd.Derivatives / r);
}

VWD pow(const VWD& vwd, Real r)
{
   REPORT
   Real v = pow(vwd.Value, r-1);
   return VWD(vwd.PS, v * vwd.Value, (r * v) * vwd.Derivatives);
}

VWD pow(const VWD& vwd, int r)
{
   if (r == 0) { REPORT  return VWD(vwd.PS, 0) + 1.0; }
   REPORT
   Real v = pow(vwd.Value, r-1);
   return VWD(vwd.PS, v * vwd.Value, (r * v) * vwd.Derivatives);
}

VWD exp(const VWD& vwd)
{
   REPORT
   Real ev = exp(vwd.Value);
   return VWD(vwd.PS, ev, ev * vwd.Derivatives);
}

VWD log(const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, log(vwd.Value), vwd.Derivatives / vwd.Value);
}

VWD sin(const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, sin(vwd.Value), cos(vwd.Value) * vwd.Derivatives);
}

VWD cos(const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, cos(vwd.Value), (-sin(vwd.Value)) * vwd.Derivatives);
}

VWD tan(const VWD& vwd)
{
   REPORT
   return VWD(vwd.PS, tan(vwd.Value), vwd.Derivatives / square(cos(vwd.Value)));
}


#ifdef WANT_ERF
// code provided by Tomas Gonzalez-Llarena
// assumes erf(double) is available as a library function
VWD erf(const VWD& vwd)
{
  REPORT
  Real e2 = exp(-square(vwd.Value));
  return VWD(vwd.PS, erf(vwd.Value), 2.0 * vwd.Derivatives * e2 / sqrt(M_PI));
}
#endif



// Gaussian numerical integration with 32 terms
VWD GaussianIntegration32(VWDOfReal& f, Real Lower, Real Upper)
{
   REPORT
   double x[] =
   {
      0.048307665687738316235,
      0.144471961582796493485,
      0.239287362252137074545,
      0.331868602282127649780,
      0.421351276130635345364,
      0.506899908932229390024,
      0.587715757240762329041,
      0.663044266930215200975,
      0.732182118740289680387,
      0.794483795967942406963,
      0.849367613732569970134,
      0.896321155766052123965,
      0.934906075937739689171,
      0.964762255587506430774,
      0.985611511545268335400,
      0.997263861849481563545
   };

   double w[] =
   {
      0.096540088514727800567,
      0.095638720079274859419,
      0.093844399080804565639,
      0.091173878695763884713,
      0.087652093004403811143,
      0.083311924226946755222,
      0.078193895787070306472,
      0.072345794108848506225,
      0.065822222776361846838,
      0.058684093478535547145,
      0.050998059262376176196,
      0.042835898022226680657,
      0.034273862913021433103,
      0.025392065309262059456,
      0.016274394730905670605,
      0.007018610009470096600
   };
   Real delta = 0.5 * (Upper - Lower); Real centre = 0.5 * (Upper + Lower);
   VWD value(0);

   for (int i = 0; i < 16; i++)
   {
      Real dev = delta * x[i];
      value += w[i] * (f(centre + dev) + f(centre - dev));
   }
   return value * delta;
}

// ****************** second derivatives ***********************************

ParameterSet VWD2::PS_null;                           // static variable

VWD2::VWD2()
   : Value(0), Derivatives(PS_null.Size()),
      SecondDerivatives(PS_null.Size(), PS_null.Size()), PS(PS_null)
{ REPORT  Derivatives = 0; SecondDerivatives = 0; }

VWD2::VWD2(Real v)
   : Value(v), Derivatives(PS_null.Size()),
      SecondDerivatives(PS_null.Size(), PS_null.Size()), PS(PS_null)
{ REPORT  Derivatives = 0; SecondDerivatives = 0; }

VWD2::VWD2(const ParameterSet& ps, Real v)
   : Value(v), Derivatives(ps.Size()),
      SecondDerivatives(ps.Size(), ps.Size()), PS(ps)
{ REPORT  Derivatives = 0; SecondDerivatives = 0; }

VWD2::VWD2(const ParameterSet& ps, Real v,
   const RowVector& d, const Matrix& d2)
   : Value(v), Derivatives(d), SecondDerivatives(d2), PS(ps)
{
   REPORT
   if (PS.Size() != d.Ncols()) Throw(Logic_error("Incompatible dimensions\n"));
}

VWD2::VWD2(const ParameterSet& ps, Real v, int k)
   : Value(v), Derivatives(ps.Size()),
      SecondDerivatives(ps.Size(), ps.Size()), PS(ps)
   { REPORT  Derivatives = 0; Derivatives(k) = 1; SecondDerivatives = 0; }

VWD2::VWD2(const ParameterSet& ps, Real v, const String& name)
   : Value(v), Derivatives(ps.Size()),
      SecondDerivatives(ps.Size(), ps.Size()), PS(ps)
{
   REPORT
   Derivatives = 0; SecondDerivatives = 0;
   int k = PS.LocateParameter(name);
   if (!k) Throw(Logic_error("Name not recognised\n"));
   Derivatives(k) = 1;
}

VWD2::VWD2(const VWD2& vwd)
   : Value(vwd.Value), Derivatives(vwd.Derivatives),
      SecondDerivatives(vwd.SecondDerivatives), PS(vwd.PS) { REPORT }

void VWD2::operator=(const VWD2& vwd)
{
   REPORT
   PS = vwd.PS; Value = vwd.Value; Derivatives = vwd.Derivatives;
   SecondDerivatives = vwd.SecondDerivatives;
}

void VWD2::operator+=(const VWD2& vwd)
{
   REPORT
   if (PS == PS_null)
   {
      REPORT
      PS = vwd.PS; int m = PS.Size();
      Derivatives.ReSize(m); Derivatives = 0.0;
      SecondDerivatives.ReSize(m, m); SecondDerivatives = 0.0;
   }
   AssertEqual(PS, vwd.PS);
   Value += vwd.Value; Derivatives += vwd.Derivatives;
   SecondDerivatives += vwd.SecondDerivatives;
}

void VWD2::operator-=(const VWD2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Value -= vwd.Value; Derivatives -= vwd.Derivatives;
   SecondDerivatives -= vwd.SecondDerivatives;
}

void VWD2::operator*=(const VWD2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Matrix OP = Derivatives.t() * vwd.Derivatives;
   Derivatives = vwd.Value * Derivatives + Value * vwd.Derivatives;
   SecondDerivatives = vwd.Value * SecondDerivatives
      + OP + OP.t()
      + Value * vwd.SecondDerivatives;
   Value *= vwd.Value;
}

void VWD2::operator/=(const VWD2& vwd)
{
   REPORT
   AssertEqual(PS, vwd.PS);
   Matrix OP = Derivatives.t() * vwd.Derivatives;
   OP += OP.t() + Value * vwd.SecondDerivatives;
   Value /= vwd.Value;
   OP = 2.0 * Value * vwd.Derivatives.t() * vwd.Derivatives - OP;
   Derivatives = (Derivatives - Value  * vwd.Derivatives) / vwd.Value;
   SecondDerivatives = (SecondDerivatives + OP / vwd.Value) / vwd.Value;
}

void VWD2::operator+=(Real r) { REPORT  Value += r; }

void VWD2::operator-=(Real r) { REPORT  Value -= r; }

void VWD2::operator*=(Real r)
   { REPORT  Value *= r; Derivatives *= r; SecondDerivatives *= r; }

void VWD2::operator/=(Real r)
   { REPORT  Value /= r; Derivatives /= r; SecondDerivatives /= r; }

VWD2 VWD2::operator-() const
{
   REPORT
   return VWD2(PS, -Value, -Derivatives, -SecondDerivatives);
}

VWD2 operator+(const VWD2& vwd1, const VWD2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return VWD2(vwd1.PS, vwd1.Value + vwd2.Value,
      vwd1.Derivatives + vwd2.Derivatives,
      vwd1.SecondDerivatives + vwd2.SecondDerivatives);
}

VWD2 operator-(const VWD2& vwd1, const VWD2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   return VWD2(vwd1.PS, vwd1.Value - vwd2.Value,
      vwd1.Derivatives - vwd2.Derivatives,
      vwd1.SecondDerivatives - vwd2.SecondDerivatives);
}

VWD2 operator*(const VWD2& vwd1, const VWD2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Matrix OP = vwd1.Derivatives.t() * vwd2.Derivatives;
   return VWD2(vwd1.PS, vwd1.Value * vwd2.Value,
      vwd2.Value * vwd1.Derivatives + vwd1.Value * vwd2.Derivatives,
      vwd2.Value * vwd1.SecondDerivatives
      + OP + OP.t()
      + vwd1.Value * vwd2.SecondDerivatives);
}

VWD2 operator/(const VWD2& vwd1, const VWD2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Matrix OP = vwd1.Derivatives.t() * vwd2.Derivatives;
   OP += OP.t() + vwd1.Value * vwd2.SecondDerivatives;
   Real v = vwd1.Value / vwd2.Value;
   OP = 2.0 * v * vwd2.Derivatives.t() * vwd2.Derivatives - OP;
   return VWD2( vwd1.PS, v,
      (vwd1.Derivatives - v * vwd2.Derivatives) / vwd2.Value,
      (vwd1.SecondDerivatives + OP / vwd2.Value) / vwd2.Value );
}

VWD2 pow(const VWD2& vwd1, const VWD2& vwd2)
{
   REPORT
   AssertEqual(vwd1.PS, vwd2.PS);
   Real v = pow(vwd1.Value, vwd2.Value);
   Real logv1 = log(vwd1.Value); Real r = (vwd2.Value / vwd1.Value);
   RowVector D1 = (r * vwd1.Derivatives + logv1 * vwd2.Derivatives);
   Matrix D2 = vwd1.Derivatives.t() * vwd2.Derivatives;
   D2 = (D2 + D2.t() - r * vwd1.Derivatives.t() * vwd1.Derivatives)
      / vwd1.Value
      + logv1 * vwd2.SecondDerivatives + r * vwd1.SecondDerivatives
      + D1.t() * D1;
   return VWD2( vwd1.PS, v, v * D1, v * D2 );
}

VWD2 operator+(Real r, const VWD2& vwd)
{
   REPORT
   return VWD2(vwd.PS, r + vwd.Value, vwd.Derivatives,
      vwd.SecondDerivatives);
}

VWD2 operator-(Real r, const VWD2& vwd)
{
   REPORT
   return VWD2(vwd.PS, r - vwd.Value, - vwd.Derivatives,
       - vwd.SecondDerivatives);
}

VWD2 operator*(Real r, const VWD2& vwd)
{
   REPORT
   return VWD2(vwd.PS, r * vwd.Value, r * vwd.Derivatives,
      r * vwd.SecondDerivatives);
}


VWD2 operator/(Real r, const VWD2& vwd)
{
   REPORT
   Real v = r / vwd.Value;
   return VWD2(vwd.PS, v, (- v / vwd.Value) * vwd.Derivatives,
      (2.0 / vwd.Value * vwd.Derivatives.t() * vwd.Derivatives
         - vwd.SecondDerivatives) * r / square(vwd.Value) );
}

VWD2 pow(Real r, const VWD2& vwd)
{
   REPORT
   Real v = pow(r, vwd.Value); Real logr = log(r); Real vlogr = v * logr;
   return VWD2( vwd.PS, v, vlogr * vwd.Derivatives,
      vlogr * vwd.SecondDerivatives
      + vlogr * logr * vwd.Derivatives.t() * vwd.Derivatives );
}

VWD2 operator+(const VWD2& vwd, Real r)
{
   REPORT
   return VWD2(vwd.PS, vwd.Value + r, vwd.Derivatives,
      vwd.SecondDerivatives);
}

VWD2 operator-(const VWD2& vwd, Real r)
{
   REPORT
   return VWD2(vwd.PS, vwd.Value - r, vwd.Derivatives,
      vwd.SecondDerivatives);
}

VWD2 operator*(const VWD2& vwd, Real r)
{
   REPORT
   return VWD2(vwd.PS, vwd.Value * r, vwd.Derivatives * r,
      vwd.SecondDerivatives * r);
}

VWD2 operator/(const VWD2& vwd, Real r)
{
   REPORT
   return VWD2(vwd.PS, vwd.Value / r, vwd.Derivatives / r,
      vwd.SecondDerivatives / r);
}


VWD2 pow(const VWD2& vwd, Real r)
{
   REPORT
   Real v0 = pow(vwd.Value, r - 2);
   Real v1 = v0 * vwd.Value;  Real v2 = v1 * vwd.Value;  v1 *= r;
   return
      VWD2( vwd.PS, v2, v1 * vwd.Derivatives,
      v1 * vwd.SecondDerivatives
      + r * (r - 1) * v0 * vwd.Derivatives.t() * vwd.Derivatives );
}

VWD2 pow(const VWD2& vwd, int r)
{
   if (r == 0) { REPORT  return VWD2(vwd.PS, 0.0) + 1.0; }
   if (r == 1) { REPORT  return vwd; }
   REPORT
   Real v0 = pow(vwd.Value, r - 2);
   Real v1 = v0 * vwd.Value;  Real v2 = v1 * vwd.Value;  v1 *= r;
   return
      VWD2( vwd.PS, v2, v1 * vwd.Derivatives,
      v1 * vwd.SecondDerivatives
      + r * (r - 1) * v0 * vwd.Derivatives.t() * vwd.Derivatives );
}

VWD2 exp(const VWD2& vwd)
{
   REPORT
   Real ev = exp(vwd.Value);
   return VWD2( vwd.PS, ev, ev * vwd.Derivatives,
      ev * (vwd.SecondDerivatives + vwd.Derivatives.t() * vwd.Derivatives) );
}

VWD2 log(const VWD2& vwd)
{
   REPORT
   return VWD2(vwd.PS, log(vwd.Value),
      vwd.Derivatives / vwd.Value,
      vwd.SecondDerivatives / vwd.Value
         - vwd.Derivatives.t() * (vwd.Derivatives / square(vwd.Value)) );
}

VWD2 sin(const VWD2& vwd)
{
   REPORT
   Real sv = sin(vwd.Value); Real cv = cos(vwd.Value);
   return VWD2( vwd.PS, sv, cv * vwd.Derivatives,
      cv * vwd.SecondDerivatives - sv * vwd.Derivatives.t() * vwd.Derivatives );
}

VWD2 cos(const VWD2& vwd)
{
   REPORT
   Real sv = sin(vwd.Value); Real cv = cos(vwd.Value);
   return VWD2(vwd.PS, cv, -sv * vwd.Derivatives,
      -sv * vwd.SecondDerivatives
      - cv * vwd.Derivatives.t() * vwd.Derivatives);
}

VWD2 tan(const VWD2& vwd)
{
   REPORT
   Real tv = tan(vwd.Value); Real cv2 = square(cos(vwd.Value));
   return VWD2(vwd.PS, tan(vwd.Value),
      vwd.Derivatives / cv2,
      vwd.SecondDerivatives / cv2
      + 2.0 * tv / cv2 * vwd.Derivatives.t() * vwd.Derivatives );
}

#ifdef WANT_ERF
// code provided by Tomas Gonzalez-Llarena
// assumes erf(double) is available as a library function
VWD2 erf(const VWD2& vwd)
{ 
  REPORT
  Real e2 = exp(-square(vwd.Value)); 
  return VWD2(vwd.PS, erf(vwd.Value),
     2.0 * vwd.Derivatives * e2 / sqrt(M_PI),
     2.0 * (vwd.SecondDerivatives
        - 2.0 * (vwd.Derivatives.t() * vwd.Derivatives ) * vwd.Value)
        * e2 / sqrt(M_PI) );
} 
#endif


// Gaussian numerical integration with 32 terms
VWD2 GaussianIntegration32(VWD2OfReal& f, Real Lower, Real Upper)
{
   REPORT
   double x[] =
   {
      0.048307665687738316235,
      0.144471961582796493485,
      0.239287362252137074545,
      0.331868602282127649780,
      0.421351276130635345364,
      0.506899908932229390024,
      0.587715757240762329041,
      0.663044266930215200975,
      0.732182118740289680387,
      0.794483795967942406963,
      0.849367613732569970134,
      0.896321155766052123965,
      0.934906075937739689171,
      0.964762255587506430774,
      0.985611511545268335400,
      0.997263861849481563545
   };

   double w[] =
   {
      0.096540088514727800567,
      0.095638720079274859419,
      0.093844399080804565639,
      0.091173878695763884713,
      0.087652093004403811143,
      0.083311924226946755222,
      0.078193895787070306472,
      0.072345794108848506225,
      0.065822222776361846838,
      0.058684093478535547145,
      0.050998059262376176196,
      0.042835898022226680657,
      0.034273862913021433103,
      0.025392065309262059456,
      0.016274394730905670605,
      0.007018610009470096600
   };
   Real delta = 0.5 * (Upper - Lower); Real centre = 0.5 * (Upper + Lower);
   VWD2 value(0);

   for (int i = 0; i < 16; i++)
   {
      Real dev = delta * x[i];
      value += w[i] * (f(centre + dev) + f(centre - dev));
   }
   return value * delta;
}


