// value with derivatives

// The "Value" is supposed to be a function of paramters t_1,...,t_n
// This structure stores the value of the "Value" and its n derivatives.


#define WANT_STREAM
#define WANT_MATH

#include "newmatio.h"
#include "rvwd.h"


// definitions for integration example
class IntEx : public VWDOfReal
{
   const VWD& A;
   const VWD& B;
public:
   IntEx(const VWD& a, const VWD& b) : A(a), B(b) {}
   VWD operator()();
};

class RIntEx : public RVWDOfReal
{
   const RVWD& A;
   const RVWD& B;
public:
   RIntEx(const RVWD& a, const RVWD& b) : A(a), B(b) {}
   RVWD operator()();
};

VWD IntEx::operator()() { return exp(A * x + B); }

RVWD RIntEx::operator()() { return exp(A * x + B); }


// test functions

template <class T>
T test1(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   return
      exp( pow(cos(Alpha), 3) / 0.25) / exp(3 * cos(pow(Alpha, 1)))
         + Gamma * Gamma * exp( pow(cos(Beta), 2) )
         * pow( exp( sin(Beta) * sin(Beta) ), -1)
         * pow(Delta, 0) + sin(-Delta) * tan(Alpha);
}

template <class T>
T test2(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   T x;
   x = sin(-Delta);
   x *= sin(Alpha);
   x /= cos(Alpha);
   x += exp(cos(3 * Alpha)) + exp(cos(Beta * 2) + 2 * log(Gamma));
   return x;
}

template <class T>
T test3(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   T x = pow(Alpha, Beta);
   x = -x;
   x -= Gamma * pow(2.194463, Delta);
   x -= Alpha * pow(Gamma, 1.54892);
   x *= -2.5;
   x -= 2.3;
   return x;
}

template <class T>
T test4(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   return
      ( exp(log(Alpha) * Beta) + Gamma * exp(Delta * log(2.194463))
         + Alpha * exp(log(Gamma) * 1.54892) ) / 0.4 - 2.3;
}

template <class T>
T test5(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   T X;
   X = Alpha;
   X += 2.35;
   X -= Beta * pow(Gamma, 2);
   X *= 1.0 / Delta;
   X /= 1.45612;
   X -= 1.289;
   X /= Alpha + Beta;
   X *= 0.8901;
   return X;
}

template <class T>
T test6(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   return
      ((Alpha + 2.35 - Beta * Gamma * Gamma) / (Delta * 1.45612) - 1.289)
         / (Alpha + Beta) * 0.8901;
}


template <class T>
T test7(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   T X = 3.33 + pow(Alpha,3);
   T Y = 2.09 * pow(Beta,3);
   T Z = 1.89 / pow(Gamma,2);
   T U = 7.83 - pow(Delta,3);
   return X * Y * Z * U;

}


template <class T>
T test8(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   T Alpha3 = Alpha * Alpha * Alpha;
   T Beta3 = Beta * Beta * Beta;
   T Gamma2 = Gamma * Gamma;
   T Delta3 = Delta * Delta * Delta;
   return
      -(Alpha3 + 3.33) * (Beta3 * 2.09) * 1.89 * (Delta3 - 7.83) / Gamma2;
}

template <class T>
T test9(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   T X = Alpha * Beta;
   T Y = Gamma / Delta;
   T Z = X + Y;
   T U = X - Y;
   return X * Y * Z * U;

}

template <class T>
T test10(const T& Alpha, const T& Beta, const T& Gamma, const T& Delta)
{
   return Alpha * Beta * Gamma / Delta
      * (pow(Alpha * Beta, 2) - pow(Gamma / Delta, 2));
}



int main()
{

   Real* s1; Real* s2; Real* s3; Real* s4;
   cout << "\nBegin test\n";   // Forces cout to allocate memory at beginning
   cout << "Now print a real number: " << 3.14159265 << endl;
   // Throw exception to set up exception buffer
   Try { Throw(Exception("Just a dummy\n")); }
   CatchAll {}
   { Matrix A1(40,200); s1 = A1.Store(); }
   { Matrix A1(1,1); s3 = A1.Store(); }

   // test ParameterSet
   Try
   {
      ParameterSet PS, PS2;
      cout << "Add P1 " << PS.AddParameter("P1") << endl;
      cout << "Add P2 " << PS.AddParameter("P2") << endl;
      cout << "Add P3 " << PS.AddParameter("P3") << endl;
      cout << "locate P3 " << PS.LocateParameter("P3") << endl;
      cout << "locate P4 " << PS.LocateParameter("P4") << endl;
      cout << "locate P2 " << PS.LocateParameter("P2") << endl;
      AssertEqual(PS, PS);
      cout << "number of parameters = " << PS.Size() << endl;
      cout << "Add P4 " << PS.AddParameter("P4") << endl;
      cout << "Add P5 " << PS.AddParameter("P5") << endl;
      cout << "Name 4 = " << PS(4) << endl;
      cout << "Name 2 = " << PS(2) << endl;
      cout << "Name 2 = " << PS(2) << endl;
      cout << "Name 3 = " << PS(3) << endl;
      cout << "Name 1 = " << PS(1) << endl;
      cout << "Name 5 = " << PS(5) << endl;
      cout << "locate P3 " << PS.LocateParameter("P3") << endl;
      cout << "locate P4 " << PS.LocateParameter("P4") << endl;
      cout << "locate P2 " << PS.LocateParameter("P2") << endl;
      cout << "number of parameters = " << PS.Size() << endl;
      cout << endl;
      cout << "This is a test of AssertEqual - should throw exception" << endl;
      AssertEqual(PS, PS2);
      cout << "finishing test" << endl;
   }
   Catch(Logic_error) { cout << Exception::what() << endl; }
   CatchAndThrow;



   // test the automatic differentiation
   Try
   {
      ParameterSet PS;
      PS.AddParameter("alpha");
      PS.AddParameter("beta");
      PS.AddParameter("gamma");
      PS.AddParameter("delta");

      Real alpha = 1.237;
      Real beta = 0.973;
      Real gamma = 3.562;
      Real delta = 2.673;

      VWD Alpha1(PS, alpha, "alpha");
      VWD Beta1(PS, beta, "beta");
      VWD Gamma1(PS, gamma, "gamma");
      VWD Delta1(PS, delta, "delta");

      VWD2 Alpha2(PS, alpha, "alpha");
      VWD2 Beta2(PS, beta, "beta");
      VWD2 Gamma2(PS, gamma, "gamma");
      VWD2 Delta2(PS, delta, "delta");

      Real x; VWD X1(PS, 0); VWD2 X2(PS, 0);

      cout << "Doing test1" << endl;
      x = test1(alpha, beta, gamma, delta);
      X1 = test1(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test1(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test2" << endl;
      x = test2(alpha, beta, gamma, delta);
      X1 = test2(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test2(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test3" << endl;
      x = test3(alpha, beta, gamma, delta);
      X1 = test3(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test3(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test4" << endl;
      x = test4(alpha, beta, gamma, delta);
      X1 = test4(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test4(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test5" << endl;
      x = test5(alpha, beta, gamma, delta);
      X1 = test5(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test5(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test6" << endl;
      x = test6(alpha, beta, gamma, delta);
      X1 = test6(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test6(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test7" << endl;
      x = test7(alpha, beta, gamma, delta);
      X1 = test7(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test7(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test8" << endl;
      x = test8(alpha, beta, gamma, delta);
      X1 = test8(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test8(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test9" << endl;
      x = test9(alpha, beta, gamma, delta);
      X1 = test9(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test9(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

      cout << "Doing test10" << endl;
      x = test10(alpha, beta, gamma, delta);
      X1 = test10(Alpha1, Beta1, Gamma1, Delta1);
      X2 = test10(Alpha2, Beta2, Gamma2, Delta2);
      cout << setw(17) << setprecision(10) << x << endl;
      cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetValue() << endl;
      cout << setw(17) << setprecision(10) << X2.GetDerivatives() << endl;
      cout << setw(17) << setprecision(10) << X2.GetSecondDerivatives() << endl;
      cout << endl;

   }
   CatchAll { cout << Exception::what() << endl; }


   // try reverse AD
   cout << "Testing reverse automatic differentiation" << endl << endl;
   Try
   {
      ParameterSet PS;
      PS.AddParameter("alpha");
      PS.AddParameter("beta");
      PS.AddParameter("gamma");
      PS.AddParameter("delta");

      Real alpha = 1.237;
      Real beta = 0.973;
      Real gamma = 3.562;
      Real delta = 2.673;

      TAPE T(PS,1024,8);

      RVWD AlphaR(T, alpha, "alpha");
      RVWD BetaR(T, beta, "beta");
      RVWD GammaR(T, gamma, "gamma");
      RVWD DeltaR(T, delta, "delta");

      {

         RVWD x = 3 * AlphaR + BetaR + 2 * GammaR;

         T.dump();

         cout << "Tape N value         = " << T.Get_N() << endl;
         cout << "Tape N_largest value = " << T.Get_N_largest() << endl;
         cout << "Tape M value         = " << T.Get_M() << endl;
         cout << endl;

         VWD y = x;

         cout << y.GetValue() << endl;
         cout << setw(17) << setprecision(10) << y.GetDerivatives() << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test1" << endl;
         VWD X1 = VWD(test1(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test2" << endl;
         VWD X1 = VWD(test2(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test3" << endl;
         VWD X1 = VWD(test3(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test4" << endl;
         VWD X1 = VWD(test4(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test5" << endl;
         VWD X1 = VWD(test5(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test6" << endl;
         VWD X1 = VWD(test6(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test7" << endl;
         VWD X1 = VWD(test7(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test8" << endl;
         VWD X1 = VWD(test8(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test9" << endl;
         VWD X1 = VWD(test9(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test10" << endl;
         VWD X1 = VWD(test10(AlphaR, BetaR, GammaR, DeltaR));
         cout << setw(17) << setprecision(10) << X1.GetValue() << endl;
         cout << setw(17) << setprecision(10) << X1.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test9X" << endl;
         VWD X = AlphaR * BetaR;
         VWD Y = GammaR / DeltaR;
         RVWD Z(T, X + Y);
         RVWD U(T, X - Y);

         VWD P = X * Y * Z * U;
         cout << setw(17) << setprecision(10) << P.GetValue() << endl;
         cout << setw(17) << setprecision(10) << P.GetDerivatives() << endl;
         cout << endl;
      }

      {
         T.purge();
         if (T.Get_N() != 5) cout << "error: N = " << T.Get_N() << endl;
         cout << "Doing test9Y" << endl;
         VWD ZERO(PS, 0); // ZERO;
         RVWD P(T, ZERO);
         {
            RVWD X = AlphaR * BetaR;
            RVWD Y = GammaR / DeltaR;
            RVWD Z = X + Y;
            RVWD U = X - Y;
            P += Y * Z * U;
         }
         cout << "Tape N value before purge = " << T.Get_N() << endl;
         T.purge(P);
         cout << "Tape N value after purge  = " << T.Get_N() << endl;
         VWD FP = P * AlphaR * BetaR;
         cout << setw(17) << setprecision(10) << FP.GetValue() << endl;
         cout << setw(17) << setprecision(10) << FP.GetDerivatives() << endl;
         cout << endl;
      }


      cout << "Tape N value         = " << T.Get_N() << endl;
      cout << "Tape N_largest value = " << T.Get_N_largest() << endl;
      cout << "Tape M value         = " << T.Get_M() << endl;
      cout << endl;

   }
   CatchAll { cout << Exception::what() << endl; }


   // Griewank P 5 example
   Try
   {
      Real x1 = 1.5, x2 = 0.5;
      ParameterSet PS;
      PS.AddParameter("x1");
      PS.AddParameter("x2");
      VWD X1(PS, x1, "x1");
      VWD X2(PS, x2, "x2");

      // Forward mode
      {
         VWD V1 = X1/X2;
         VWD V4 = V1 - exp(X2);
         VWD YF = (sin(V1) + V4) * V4;
         cout << "Griewank P5 - forward method" << endl;
         cout << "value:        "
            << setw(10) << setprecision(5) << YF.GetValue() << endl;
         cout << "derivatives:  "
            << setw(10) << setprecision(5) << YF.GetDerivatives() << endl;
      }

      // Reverse mode
      {
         TAPE T(PS,20,2);
         // cout << "Throw exception" << endl;
         // Throw(Exception("Test of exception\n"));
         RVWD RX1(T,X1);  RVWD RX2(T,X2);
         {
            // go to new block to make sure all temporaries are deleted
            // when we try reset
            RVWD RV1 = RX1/RX2;
            RVWD RV4 = RV1 - exp(RX2);
            VWD YR = (sin(RV1) + RV4) * RV4;
            cout << "Griewank P5 - reverse method" << endl;
            cout << "value:        "
               << setw(10) << setprecision(5) << YR.GetValue() << endl;
               cout << "derivatives:  "
               << setw(10) << setprecision(5) << YR.GetDerivatives() << endl;
            cout << "Number of tape locations used = " << T.Get_N() << endl;
            cout << endl;
            // Throw(Exception("Test of exception\n"));
            T.dump();
            RX1 = RVWD(T, 0.0);
            RX2 = RVWD(T, 0.0);
            RV1 = RVWD(T, 0.0);
            RV4 = RVWD(T, 0.0);
            T.dump();
         }
         T.dump();
         T.reset();
         T.dump();
      }

      // Reverse mode with purge
      {
         TAPE T(PS,20,2);
         RVWD YR(T,5);
         {
            RVWD RX1(T,X1);  RVWD RX2(T,X2);
            RVWD RV1 = RX1/RX2;
            RVWD RV4 = RV1 - exp(RX2);
            YR += (sin(RV1) + RV4) * RV4;
            YR -= 5;
         }
         T.purge(YR);
         VWD YRF = YR;
         cout << "Griewank P5 - reverse method with purge" << endl;
         cout << "value:        "
            << setw(10) << setprecision(5) << YRF.GetValue() << endl;
         cout << "derivatives:  "
            << setw(10) << setprecision(5) << YRF.GetDerivatives() << endl;
         cout << "Number of tape locations used = " << T.Get_N() << endl;
         cout << endl;
         T.dump();
      }

   }
   CatchAll { cout << Exception::what() << endl; }


   // Integration example
   // Integrate exp(A x + B) wrt x between 1 and 2 and find derivatives
   // wrt A and B.
   // Result should be {exp(2A+B)-exp(A+B)}/A
   // Derivs should be {2exp(2A+B)-exp(A+B)}/A - {exp(2A+B)-exp(A+B)}/A**2
   //    and {exp(2A+B)-exp(A+B)}/A
   Try
   {
      ParameterSet PS;
      PS.AddParameter("A");
      PS.AddParameter("B");
      Real a = 0.754; Real b = 0.239;  // where we are going to evaluate derivs
      VWD A(PS, a, "A");
      VWD B(PS, b, "B");
      IntEx Integrand(A, B);           // function we want to integrate
      VWD Result = GaussianIntegration32(Integrand, 1.0, 2.0);
      cout << "Integration example using forward differentiation" << endl;
      cout << endl;
      cout << "value:        "
         << setw(10) << setprecision(5) << Result.GetValue() << endl;
      cout << "derivatives:  "
         << setw(10) << setprecision(5) << Result.GetDerivatives() << endl;
      Real t1 = (exp(2 * a + b) - exp(a + b)) / a;
      Real t2 = (2 * exp(2 * a + b) - exp(a + b)) / a;
      cout << "Should be" << endl;
      cout << "value:        "
         << setw(10) << setprecision(5) << t1 << endl;
      cout << "derivatives:  "
         << setw(10) << setprecision(5) << (t2 - t1 / a) << " "
         << setw(10) << setprecision(5) << t1 << endl;
      cout << endl;

   }
   CatchAll { cout << Exception::what() << endl; }

   // same thing using reverse AD
   Try
   {
      ParameterSet PS;
      PS.AddParameter("A");
      PS.AddParameter("B");
      TAPE Tape(PS, 10000, 2);
      Real a = 0.754; Real b = 0.239;  // where we are going to evaluate derivs
      RVWD A(Tape, a, "A");
      RVWD B(Tape, b, "B");
      RIntEx Integrand(A, B);           // function we want to integrate
      RVWD R_Result = GaussianIntegration32(Integrand, 1.0, 2.0);
      cout << "Integration example using reverse differentiation" << endl;
      cout << "Number of tape locations before purge = "
         << Tape.Get_N() << endl;
      Tape.purge(R_Result);
      cout << "Number of tape locations after purge  = "
         << Tape.Get_N() << endl;
      cout << endl;
      VWD Result = R_Result;
      cout << "value:        "
         << setw(10) << setprecision(5) << Result.GetValue() << endl;
      cout << "derivatives:  "
         << setw(10) << setprecision(5) << Result.GetDerivatives() << endl;
      Real t1 = (exp(2 * a + b) - exp(a + b)) / a;
      Real t2 = (2 * exp(2 * a + b) - exp(a + b)) / a;
      cout << "Should be" << endl;
      cout << "value:        "
         << setw(10) << setprecision(5) << t1 << endl;
      cout << "derivatives:  "
         << setw(10) << setprecision(5) << (t2 - t1 / a) << " "
         << setw(10) << setprecision(5) << t1 << endl;
      cout << endl;

   }
   CatchAll { cout << Exception::what() << endl; }


   // check for lost memory
   { Matrix A1(40,200); s2 = A1.Store(); }
   cout << "\n(The following memory checks are probably not valid with all\n";
   cout << "compilers - see documentation)\n";
   cout << "\nChecking for lost memory (large block): "
      << (unsigned long)s1 << " " << (unsigned long)s2 << " ";
   if (s1 != s2) cout << " - see Newmat section 2.8\n\n";
   else cout << " - ok\n\n";
   { Matrix A1(1,1); s4 = A1.Store(); }
   cout << "\nChecking for lost memory (small block): "
      << (unsigned long)s3 << " " << (unsigned long)s4 << " ";
   if (s3 != s4) cout << " - see Newmat section 2.8\n\n";
   else cout << " - ok\n\n";



   return 0;
}
