// value with derivatives - reverse mode

// The "Value" is supposed to be a function of parameters t_1,...,t_n
// This structure stores the value of the "Value" and its n derivatives.


#define WANT_STREAM
#define WANT_MATH

#include "rvwd.h"
#include "newmatio.h"

#ifdef DO_REPORT
#define REPORT { static ExeCounter ExeCount(__LINE__,11); ++ExeCount; }
#else
#define REPORT {}
#endif



void TAPE_Class::DecrRef()
{
   REPORT
   if (--RefCount < 0) { REPORT  delete this; }
}

TAPE::TAPE(const ParameterSet& ps, int n, int m)
{
   REPORT
   TC = new TAPE_Class(ps, n, m);
   if (!TC) Throw(Bad_alloc());
}

void TAPE::operator=(const TAPE& t)
{
   REPORT
   TAPE_Class* TC1 = TC; TC = t.TC;
   TC->IncrRef(); TC1->DecrRef();
}

TAPE::~TAPE() { TC->DecrRef(); }

TAPE_Class::TAPE_Class(const ParameterSet& ps, int n, int m)
   : RefCount(0), PS(ps), N_max(n), N(0), N_largest(0), M_max(m), M(0)
{
   // delay resizing RV matrix until we need to
   TE = new TapeElement[N_max]; if (!TE) Throw(Bad_alloc());
   push(0, 0.0, true);    // reference for zero or constant RVWD
   TE[0].ref_count = 1;   // so it will never be purged
}

TAPE_Class::~TAPE_Class()
{
   // cout << "deleting tape" << endl;
   delete [] TE;
}

void TAPE_Class::push(int l, Real w, bool f)
{
   if (N == N_max) Throw(Runtime_error("Tape full"));
   TapeElement& TEN = TE[N++];
   TEN.location = l; TEN.weight = w; TEN.first = f; TEN.ref_count = 0;
}

void TAPE_Class::dump() const
{
   cout << "Tape dump - elements" << endl;
   for (int i = 0; i < N; i++)
   {
      TapeElement te = TE[i];
      cout << i << "  " << te.location << "  " << te.weight;
      cout << "  " << te.cum_weight << "  " << te.ref_count;
      if (te.first) cout << "  *";
      cout << endl;
   }
   cout << endl;
   cout << "Tape dump - vectors" << endl;
   if (M > 0) cout << setprecision(4) << setw(8) << RV.Rows(1,M) << endl;
   cout << endl;
}

void TAPE_Class::reset()
{
   for (int i = 1; i < N; i++)
      if (TE[i].ref_count != 0)
         Throw(Logic_error("Can't reset - active RVWDs"));
   N = 1; M = 0;
}

void TAPE_Class::purge()
{
   if (N_largest < N) N_largest = N;
   for (int i = N - 1; i >= 0; --i)
   {
      if (TE[i].ref_count != 0) { N = i + 1; return; }
   }
   N = 0;
}

// check that rvwd is at end of TAPE then consolidate references to it
// by doing a mini reverse_sweep

void TAPE_Class::purge(RVWD& rvwd)
{
   int i, j; bool f = true;
   purge();
   if (rvwd.location != N-1) return;       // rvwd not at end of list
   int rc = TE[N-1].ref_count;
   TE[N-1].ref_count = 0;                  // to keep ref_counts on tape = 0
   int N1 = N;                             // current next element of tape
   // locate the tape elements that we can consolidate
   int N0 = 0;                             // revised TE will start here
   for (i = N-2; i >= 0; --i)
      if (TE[i].ref_count != 0) { N0 = i + 1; break; }
   // do the consolidation
   for (i = N0; i < N; ++i) TE[i].cum_weight = 0.0;
   for (i = N-1; ; --i)
      { TE[i].cum_weight = TE[i].weight; if (TE[i].first) break; }
   for (i = N-1; i >= N0; --i)
   {
      const TapeElement& te = TE[i]; int tel = te.location;
      Real tecw = te.cum_weight;
      if (tecw != 0)                       // ignore if no weight
         {
         if (tel >= N0)                       // in consolidation range
         {
            for (j = tel; ; --j)
            {
               TE[j].cum_weight += tecw * TE[j].weight;
               if (TE[j].first) break;
            }
         }
         else                                 // outside consolidation range
         {
            bool found = false;
            for (j = N1; j < N; ++j)          // see if new element
               if (tel == TE[j].location)
                  { TE[j].weight += tecw; found = true; break; }
            if (!found) { push(tel, tecw, f); f = false; }
         }
      }
   }
   if (N_largest < N) N_largest = N;
   // copy the new tape elements
   for (i = N1; i < N; ++i) TE[N0++] = TE[i];
   N = N0; rvwd.location = N-1;
   TE[N-1].ref_count = rc;               // restore ref_count
}

RowVector TAPE_Class::reverse_sweep(int loc)
{
   if (loc >= N) Throw(Logic_error("Invalid location in reverse_sweep"));
   int i, j;
   RowVector Sum(PS.Size()); Sum = 0.0;
   for (i = 0; i <= loc; ++i) TE[i].cum_weight = 0.0;
   for (i = loc; ; --i)
      { TE[i].cum_weight = TE[i].weight; if (TE[i].first) break; }
   for (i = loc; i >= 0; --i)
   {
      const TapeElement& te = TE[i]; int tel = te.location;
      if (tel > 0)        // omit tel == 0 corresponds to zero element
      {
         for (j = tel; ; --j)
         {
            TE[j].cum_weight += te.cum_weight * TE[j].weight;
            if (TE[j].first) break;
         }
      }
      else if (tel < 0) Sum += te.cum_weight * RV.Row(-tel);
   }
   return Sum;
}

int TAPE_Class::Get_N_largest() const     // the largest N has been
{
   return N > N_largest ? N : N_largest;
}


void AssertEqual(const TAPE& t1, const TAPE& t2)
{
   REPORT
   if (t1.TC != t2.TC) Throw(Logic_error("Different parameter sets\n"));
}

inline Real square(Real x) { return x * x; }

// see if there is a slot in RV that is not referenced
// could be more efficient?
int TAPE_Class::FindSpareSlot()
{
   int i;
   bool* spare = new bool[M_max]; if (!spare) Throw(Bad_alloc());
   for (i = 0; i < M_max; ++i) spare[i] = true;
   for (i = 0; i < N; ++i)
   {
      int tel = TE[i].location;
      if (tel < 0) spare[-tel-1] = false;
   }
   for (i = 0; i < M_max; ++i) if (spare[i])
      { delete [] spare; return i+1; }
   delete [] spare; return 0;
}

int TAPE_Class::IncludeDerivatives(const RowVector& rv)
{
   if (M == 0) { RV.ReSize(M_max,PS.Size()); RV = -1; }
   if (M >= M_max)
   {
      int i = FindSpareSlot();
      if (i > 0) { RV.Row(i) = rv; return i; }
      else Throw(Runtime_error("VWD space in TAPE is full"));
   }
   RV.Row(++M) = rv; return M;
}

ParameterSet RVWD::PS_null;
TAPE RVWD::TAPE_null(PS_null,1,0);


// The following 4 functions manipulate the location variable in a RVWD
// This points to the location on a tape where the current weight values
// of the RWVD are held.
// As well as updating the location they must adjust the reference counters
// on the tape so the tape knows which variables can still be refered to.

// SetLocation() refers to constructors so the new reference counter needs to be
// set to 1, but there is no old reference counter to be adjusted

// UpdateLocation() is when a variable is being given a new value so a new
// location is being assigned on the tape and the old location has to have
// the reference counter decremented and the new location has to have the
// reference counter set to 1.

// UpdateLocation(t) is when a variable is being given a new value so a new
// location is being assigned on the tape and the old location has to have
// the reference counter decremented and the new location has to have the
// reference counter set to 1. Possibly the tape has changed so we need to be
// sure the counters on the correct tapes are being adjusted.

// UpdateLocation(t,l) is used by RVWD::operator= where the RVWD is now refering
// to a location already in use

void RVWD::SetLocation()
{
   REPORT
   location = Tape.Get_N() - 1;
   Tape.TC->TE[location].ref_count = 1;
}

void RVWD::UpdateLocation()
{
   REPORT
   --Tape.TC->TE[location].ref_count;
   location = Tape.Get_N() - 1;
   Tape.TC->TE[location].ref_count = 1;
}

void RVWD::UpdateLocation(const TAPE& t)
{
   REPORT
   --Tape.TC->TE[location].ref_count;
   Tape = t; location = Tape.Get_N() - 1;
   Tape.TC->TE[location].ref_count = 1;
}

void RVWD::UpdateLocation(const TAPE& t, int l)
{
   REPORT
   --Tape.TC->TE[location].ref_count;
   Tape = t; location = l;
   ++Tape.TC->TE[location].ref_count;
}

// RVWD with given parameters and location
RVWD::RVWD(int l, const TAPE& t, Real v)
  : Tape(t), Value(v), location(l)
  { ++Tape.TC->TE[location].ref_count; }


// default constructor
RVWD::RVWD()
  : Tape(TAPE_null), Value(0), location(0)
  { ++Tape.TC->TE[location].ref_count; }

RVWD::RVWD(Real v)
  : Tape(TAPE_null), Value(v), location(0)
  { ++Tape.TC->TE[location].ref_count; }

// make RVWD with real value - derivatives are zero
RVWD::RVWD(const TAPE& t, Real v)
  : Tape(t), Value(v), location(0)
  { ++Tape.TC->TE[location].ref_count; }

// convert VWD into RVWD
RVWD::RVWD(const TAPE& t, const VWD& vwd)
   : Tape(t), Value(vwd.GetValue())
{
   int i = Tape.IncludeDerivatives(vwd.GetDerivatives());
   Tape.push(-i, 1.0, true);
   SetLocation();
}

// build RVWD directly
RVWD::RVWD(const TAPE& t, Real v, const String& name)
   : Tape(t), Value(v)
{
   REPORT
   ParameterSet ps = t.GetParameterSet();
   RowVector derivatives(ps.Size()); derivatives = 0;
   int k = ps.LocateParameter(name);
   if (!k) Throw(Logic_error("Name not recognised\n"));
   derivatives(k) = 1;
   int i = Tape.IncludeDerivatives(derivatives);
   Tape.push(-i, 1.0, true);
   SetLocation();
}

// build RVWD directly using location
RVWD::RVWD(const TAPE& t, Real v, int k)
   : Tape(t), Value(v)
{
   REPORT
   ParameterSet ps = t.GetParameterSet();
   RowVector derivatives(ps.Size()); derivatives = 0;
   derivatives(k) = 1;
   int i = Tape.IncludeDerivatives(derivatives);
   Tape.push(-i, 1.0, true);
   SetLocation();
}

// convert RVWD into VWD
VWD::VWD(const RVWD& rvwd)
   :  Value(rvwd.GetValue()),
      Derivatives(rvwd.GetTape().reverse_sweep(rvwd.GetLocation())),
      PS(rvwd.GetParameterSet())
{}

// do reverse sweep and get derivatives
ReturnMatrix RVWD::GetDerivatives() const
{
   return GetTape().reverse_sweep(GetLocation());
}

//copy constructor - just do copy
RVWD::RVWD(const RVWD& rvwd)
   : Tape(rvwd.Tape), Value(rvwd.Value), location(rvwd.location)
   { REPORT ++Tape.TC->TE[location].ref_count; }

// make a new RVWD pointing to existing one with new value and weight
RVWD::RVWD(const RVWD& rvwd, Real v, Real w)
   : Tape(rvwd.Tape), Value(v)
{
   REPORT
   Tape.push(rvwd.location, w, true);
   SetLocation();

}

// make a new RVWD pointing to existing two with new value and weights
RVWD::RVWD(const RVWD& rvwd1, const RVWD& rvwd2, Real v, Real w1, Real w2)
   : Tape(rvwd1.Tape), Value(v)
{
   REPORT
   AssertEqual(rvwd1.Tape, rvwd2.Tape);
   Tape.push(rvwd1.location, w1, true);
   Tape.push(rvwd2.location, w2, false);
   SetLocation();
}


void RVWD::operator=(const RVWD& rvwd)
{
   // use current location
   REPORT
   Value = rvwd.Value;
   UpdateLocation(rvwd.Tape, rvwd.location);
}

void RVWD::operator+=(const RVWD& rvwd)
{
   REPORT
   if (Tape.TC == TAPE_null.TC)
      {  REPORT Tape = rvwd.Tape; ++Tape.TC->TE[location].ref_count; }
   AssertEqual(Tape, rvwd.Tape);
   Value += rvwd.Value;
   Tape.push(location, 1.0, true);
   Tape.push(rvwd.location, 1.0);
   UpdateLocation();
}

void RVWD::operator-=(const RVWD& rvwd)
{
   REPORT
   AssertEqual(Tape, rvwd.Tape);
   Value -= rvwd.Value;
   Tape.push(location, 1.0, true);
   Tape.push(rvwd.location, -1.0);
   UpdateLocation();
}

void RVWD::operator*=(const RVWD& rvwd)
{
   REPORT
   AssertEqual(Tape, rvwd.Tape);
   Tape.push(location, rvwd.Value, true);
   Tape.push(rvwd.location, Value);
   Value *= rvwd.Value;
   UpdateLocation();
}

void RVWD::operator/=(const RVWD& rvwd)
{
   REPORT
   AssertEqual(Tape, rvwd.Tape);
   Value /= rvwd.Value;
   Tape.push(location, 1.0 / rvwd.Value, true);
   Tape.push(rvwd.location, -Value / rvwd.Value);
   UpdateLocation();
}

void RVWD::operator*=(Real r)
{
   REPORT
   Value *= r;
   Tape.push(location, r, true);
   UpdateLocation();
}

void RVWD::operator/=(Real r)
{
   REPORT
   Value /= r;
   Tape.push(location, 1.0 / r, true);
   UpdateLocation();
}

RVWD RVWD::operator-() const
   { REPORT  return RVWD(*this, -Value, -1.0); }

RVWD operator+(const RVWD& rvwd1, const RVWD& rvwd2)
{
   REPORT
   return RVWD(rvwd1, rvwd2, rvwd1.Value + rvwd2.Value, 1.0, 1.0);
}

RVWD operator-(const RVWD& rvwd1, const RVWD& rvwd2)
{
   REPORT
   return RVWD(rvwd1, rvwd2, rvwd1.Value - rvwd2.Value, 1.0, -1.0);
}

RVWD operator*(const RVWD& rvwd1, const RVWD& rvwd2)
{
   REPORT
   return RVWD(rvwd1, rvwd2, rvwd1.Value * rvwd2.Value,
      rvwd2.Value, rvwd1.Value);
}

RVWD operator/(const RVWD& rvwd1,
   const RVWD& rvwd2)
{
   REPORT
   Real v = rvwd1.Value / rvwd2.Value;
   return RVWD( rvwd1, rvwd2, v, 1.0 / rvwd2.Value, -v / rvwd2.Value );
}

RVWD pow(const RVWD& rvwd1,
   const RVWD& rvwd2)
{
   REPORT
   Real v = pow(rvwd1.Value, rvwd2.Value);
   return RVWD(rvwd1, rvwd2, v,
      v * rvwd2.Value / rvwd1.Value, v * log(rvwd1.Value) );
}

RVWD operator+(Real r, const RVWD& rvwd)
{
   // use current location
   REPORT
   return RVWD(rvwd.location, rvwd.Tape, r + rvwd.Value);
}

RVWD operator-(Real r, const RVWD& rvwd)
{
   REPORT
   return RVWD(rvwd, r - rvwd.Value, -1.0);
}

RVWD operator*(Real r, const RVWD& rvwd)
{
   REPORT
   return RVWD(rvwd, r * rvwd.Value, r);
}

RVWD operator/(Real r, const RVWD& rvwd)
{
   REPORT
   Real v = r / rvwd.Value;
   return RVWD(rvwd, v, - v / rvwd.Value);
}

RVWD pow(Real r, const RVWD& rvwd)
{
   REPORT
   Real v = pow(r, rvwd.Value);
   return RVWD(rvwd, v, v * log(r));
}

RVWD operator+(const RVWD& rvwd, Real r)
{
   // use current location
   REPORT
   return RVWD(rvwd.location, rvwd.Tape, rvwd.Value + r);
}

RVWD operator-(const RVWD& rvwd, Real r)
{
   // use current location
   REPORT
   return RVWD(rvwd.location, rvwd.Tape, rvwd.Value - r);
}

RVWD operator*(const RVWD& rvwd, Real r)
{
   REPORT
   return RVWD(rvwd, rvwd.Value * r, r);
}

RVWD operator/(const RVWD& rvwd, Real r)
{
   REPORT
   return RVWD(rvwd, rvwd.Value / r, 1.0 / r);
}

RVWD pow(const RVWD& rvwd, Real r)
{
   REPORT
   Real v = pow(rvwd.Value, r-1);
   return RVWD(rvwd, v * rvwd.Value, r * v);
}

RVWD pow(const RVWD& rvwd, int r)
{
   if (r == 0) { REPORT  return RVWD(rvwd, 1.0, 0.0); }
   REPORT
   Real v = pow(rvwd.Value, r-1);
   return RVWD(rvwd, v * rvwd.Value, r * v);
}

RVWD exp(const RVWD& rvwd)
{
   REPORT
   Real ev = exp(rvwd.Value); return RVWD(rvwd, ev, ev);
}

RVWD log(const RVWD& rvwd)
{
   REPORT
   return RVWD(rvwd, log(rvwd.Value), 1.0 / rvwd.Value);
}

RVWD sin(const RVWD& rvwd)
{
   REPORT
   return RVWD(rvwd, sin(rvwd.Value), cos(rvwd.Value));
}

RVWD cos(const RVWD& rvwd)
{
   REPORT
   return RVWD(rvwd, cos(rvwd.Value), -sin(rvwd.Value));
}

RVWD tan(const RVWD& rvwd)
{
   REPORT
   return RVWD(rvwd, tan(rvwd.Value), 1.0 / square(cos(rvwd.Value)));
}


#ifdef WANT_ERF
// code provided by Tomas Gonzalez-Llarena
// assumes erf(double) is available as a library function
RVWD erf(const RVWD& rvwd)
{
  REPORT
  Real e2 = exp(-square(rvwd.Value));
  return RVWD(rvwd, erf(rvwd.Value), 2.0 * e2 / sqrt(M_PI));
}
#endif

// Gaussian numerical integration with 32 terms
RVWD GaussianIntegration32(RVWDOfReal& f, Real Lower, Real Upper)
{
   REPORT
   double x[] =
   {
      0.048307665687738316235,
      0.144471961582796493485,
      0.239287362252137074545,
      0.331868602282127649780,
      0.421351276130635345364,
      0.506899908932229390024,
      0.587715757240762329041,
      0.663044266930215200975,
      0.732182118740289680387,
      0.794483795967942406963,
      0.849367613732569970134,
      0.896321155766052123965,
      0.934906075937739689171,
      0.964762255587506430774,
      0.985611511545268335400,
      0.997263861849481563545
   };

   double w[] =
   {
      0.096540088514727800567,
      0.095638720079274859419,
      0.093844399080804565639,
      0.091173878695763884713,
      0.087652093004403811143,
      0.083311924226946755222,
      0.078193895787070306472,
      0.072345794108848506225,
      0.065822222776361846838,
      0.058684093478535547145,
      0.050998059262376176196,
      0.042835898022226680657,
      0.034273862913021433103,
      0.025392065309262059456,
      0.016274394730905670605,
      0.007018610009470096600
   };
   Real delta = 0.5 * (Upper - Lower); Real centre = 0.5 * (Upper + Lower);
   RVWD value(0);

   for (int i = 0; i < 16; i++)
   {
      Real dev = delta * x[i];
      value += w[i] * (f(centre + dev) + f(centre - dev));
   }
   return value * delta;
}



