/// \ingroup gpi
///@{

/// \file plotdemo.cpp
/// Demonstration of gnuplot interface.

#define WANT_STREAM
#define WANT_MATH
#define WANT_FSTREAM


#include "include.h"
#include "str.h"
#include "newmatio.h"
#include "gnuplot.h"


#ifdef use_namespace
using namespace RBD_LIBRARIES;
#endif


/// Called by main()
int my_main()
{
   // CommandFile = "Commands.gp";  // use only if in hold mode

   #ifdef set_unix_options 
   TemporaryDirectory = "temp" ;      // for gnuplot files
   #else
   TemporaryDirectory = "c:/temp" ;   // for gnuplot files
   #endif
   cout << "Demonstration of plot program\n";

   char a_char;
   // MultiPlot MP("Test of Multiplot");     // 4.2 and higher
   MultiPlot MP;

   {
      // Construct a plot object
      Plot2 plotA("Points connected by lines; lines; labels; key");
      Matrix X(10,2);
      // load some data
      X << 1 << 3.5
        << 2 << 4.7
        << 3 << 4.1
        << 4 << 5.2
        << 5 << 5.6
        << 6 << 4.9
        << 7 << 5.3
        << 8 << 5.0
        << 9 << 4.5
        << 10 << 4.8;
      // make a line & point graph of the data
      LinePointSeries LS1("The Data", X, 3, 1);
      // set up the axes
      Axis X_axis("X values");
      Axis Y_axis("Y values");
      Y_axis.SetRange(2, 7);   // set range so lines and labels are in range
      // add the axes to the plot
      plotA.Set_X_Axis(X_axis); plotA.Set_Y_Axis(Y_axis);
      // include a key (legend) in the plot
      plotA.WantKey(true);
      // add the linepoint graph to the plot and
      // also add two horizontal lines and labels
      plotA << LS1
            << Line(1,3.0,10,3.0) << Label("LCL",1.1,2.9,Label::left)
            << Line(1,5.5,10,5.5) << Label("UCL",1.1,5.6,Label::left);
      // actually do the plot
      plotA.DoPlot();
      // add the plot to the multiplot
      MP.Size(0.5, 0.48); MP.Origin(0.0,0.48); MP.PointSize(0.75); MP << plotA;
   }
   cout << "Press <enter>";
   cin.get(a_char);

   {
      // Construct a plot object
      Plot2 plotB("Points and fitted line; key");
      Matrix X(10,2);
      // load some data
      X << 1 << 2.0
        << 2 << 3.5
        << 3 << 4.1
        << 4 << 5.0
        << 5 << 5.6
        << 6 << 5.9
        << 7 << 5.3
        << 8 << 5.0
        << 9 << 4.5
        << 10 << 4.1;
      // make a point graph of the data
      PointSeries LS1("The data", X, 3, 1);
      // fit quadratic curve - not a very intelligent way of doing this
      // but OK for the small number of points
      Matrix Xmat(10,3);
      Xmat.column(1) = 1.0;
      Xmat.column(2) = X.column(1);
      Xmat.column(3) = SP(X.column(1), X.column(1)); // squares of X values
      // least sqaures fit
      ColumnVector Beta = (Xmat.t() * Xmat).i() * (Xmat.t() * X.column(2));
      // calculate the fitted line at lots of points so we get a smooth curve
      Matrix Fit(1000,2);
      for (int i = 1; i <= 1000; ++i)
      {
         Real x = (i + 50) / 100.0;
         Fit(i, 1) = x;
         Fit(i, 2) = Beta(1) + Beta(2) * x + Beta(3) * x * x;
      }
      // make a line plot of the fitted line
      LineSeries LS2("Fitted line", Fit, 2);
      // set up the axes   
      Axis X_axis("X values");
      Axis Y_axis("Y values");
      // add the axes to the plot
      plotB.Set_X_Axis(X_axis); plotB.Set_Y_Axis(Y_axis);
      // include a key (legend) in the plot
      plotB.WantKey(true);
      // add the graphs to the plot
      plotB << LS1 << LS2;
      // actually do the plot
      plotB.DoPlot();
      // add the plot to the multiplot
      MP.Size(0.5, 0.48); MP.Origin(0.5,0.48); MP.PointSize(0.75); MP << plotB;
   }
   
   cout << "Press <enter>";
   cin.get(a_char);

   {
      Plot2 plotC("Error bars");
      Matrix X(5,4);
      // Load data including ends of error bars
      X.row(1) << 1 << .50 << .48 << .53;
      X.row(2) << 2 << .55 << .53 << .58;
      X.row(3) << 3 << .62 << .60 << .65;
      X.row(4) << 4 << .59 << .57 << .62;
      X.row(5) << 5 << .64 << .62 << .67;
      YErrorBarSeries YEBS1("ErrorBarSeries", X, 4, 1);
      Axis X_axis("value of X"); X_axis.SetRange(0.5, 5.5);
      Axis Y_axis("value of Y");
      plotC.Set_X_Axis(X_axis); plotC.Set_Y_Axis(Y_axis);
      plotC.WantKey(false);
      plotC << YEBS1;
      plotC.DoPlot();
      MP.Size(0.5, 0.48); MP.Origin(0.0,0.0); MP.PointSize(0.75); MP << plotC;
   }
   
   cout << "Press <enter>";
   cin.get(a_char);
   
   {
      Plot2 plotD("Function and point series");
      Axis x_axis("value of x");
      Axis y_axis("value of y");
      plotD.Set_X_Axis(x_axis);
      plotD.Set_Y_Axis(y_axis);
      Matrix X(25,2);
      for (int i = 1; i <= 25; i++) { X(i,1) = i; X(i,2) = i*i; }
      PointSeries PS1("Quadratic",X,4,3);

      plotD.AddFunction("x**2");
      plotD.WantKey(false);
      plotD << PS1;
      plotD.DoPlot();
      MP.Size(0.5, 0.48); MP.Origin(0.5,0.0); MP.PointSize(0.75); MP << plotD;
   }
   cout << "Press <enter>";
   cin.get(a_char);
   
   // plot the multiplot of the first four graphs   
   MP.DoPlot();
   
   cout << "Press <enter>";
   cin.get(a_char);
   
   
   {
      Plot2 plotE("XY Error bars");
      Matrix X(5,6);
      // Load data including ends of error bars
      X.row(1) << 1 << .50 << 0.9 << 1.1 << .48 << .53;
      X.row(2) << 2 << .55 << 1.9 << 2.1 << .53 << .58;
      X.row(3) << 3 << .62 << 2.9 << 3.1 << .60 << .65;
      X.row(4) << 4 << .59 << 3.9 << 4.1 << .57 << .62;
      X.row(5) << 5 << .64 << 4.9 << 5.1 << .62 << .67;
      XYErrorBarSeries XYEBS1("ErrorBarSeries", X, 4, 1);
      Axis X_axis("value of X"); X_axis.SetRange(0.5, 5.5);
      Axis Y_axis("value of Y");
      plotE.Set_X_Axis(X_axis); plotE.Set_Y_Axis(Y_axis);
      plotE.WantKey(false);
      plotE << XYEBS1;
      plotE.DoPlot();
   }
   
   cout << "Press <enter>";
   cin.get(a_char);



   {
      Plot3 plot3("Three dimensional graph");
      Matrix X(110,3);
      int ij = 0;
      for (int i = 1; i <= 10; i++)
      {
         for (int j = 1; j <= 10; j++)
         {
            ++ij;
            X(ij, 1) = i; X(ij, 2) = j;
            X(ij, 3) = i * j * j;
         }
         ++ij;
         X(ij,1) = 1.1E50;
      }
      LineSeries LS1("Plane", X, 2);
      Axis X_axis("value of X");
      Axis Y_axis("value of Y");
      Axis Z_axis("value of Z");
      plot3.Set_X_Axis(X_axis);
      plot3.Set_Y_Axis(Y_axis);
      plot3.Set_Z_Axis(Z_axis);
      plot3.WantKey(false);
      plot3 << LS1;
      // plot3.AddCommand("set contour base");
      plot3.DoPlot();
   }
   cout << "Press <enter>";
   cin.get(a_char);

   {
      Plot3 plotf("Function");
      Axis x_axis("x"); x_axis.SetRange(-3.5, 3.5);
      Axis y_axis("y"); y_axis.SetRange(-1.5, 4.5);
      Axis z_axis("z"); z_axis.SetRange(-10.5, 5.5);
      plotf.Set_X_Axis(x_axis);
      plotf.Set_Y_Axis(y_axis);
      plotf.Set_Z_Axis(z_axis);
      plotf.Set("contour surface");
      //plotf.Set("contour base");
      //plotf.Set("hidden3d");
      plotf.Set("noparametric");
      plotf.Set("isosamples 60");
      plotf.Set("samples 20");
      plotf.Set("cntrparam levels 5");
      plotf.Set("cntrparam linear");
      plotf.Set("view 20, 345, 1, 2");

      plotf.AddFunction("-log((1-x)**2 +100*(y-x**2)**2)");
      plotf.DoPlot();
   }



   return 0;
}

/// Call my_main() - use this to catch exceptions.
/// Use macros for exception names for compatibility with simulated exceptions
int main()
{
   Try  { return my_main(); }
   Catch(BaseException) { cout << BaseException::what() << "\n"; }
   CatchAll { cout << "\nProgram fails - exception generated\n\n"; }
   return 0;
}




///@}

