/// \defgroup gpi Gnuplot interface
///@{

/// \file gnuplot.h
///
/// C++ interface to Gnuplot - headers.
///
/// This stage is very low tech - write graph information to files
/// and then run gnuplot on these files.
///
/// Try to be sufficiently flexible to enable other
/// interfaces to be built.
                              
#ifndef GNUPLOT_LIB
#define GNUPLOT_LIB 0

#include "include.h"
#include "newmat.h"            // my matrix package
#include "str.h"               // String class
#include "str_fns.h"

#ifdef use_namespace
namespace RBD_LIBRARIES {
#endif


extern String TemporaryDirectory;
extern String CommandFile;

class Plot2;
class Plot3;
class MultiPlot;



/// The components such as LineSeries are derived from this.
class BaseSeries
{
   String Title;               ///< Title of a graph.
   String FileName;            ///< File where data for graph is stored.
public:
   BaseSeries(const String& title, const Matrix& data);
   BaseSeries(const BaseSeries& bs) : Title(bs.Title), FileName(bs.FileName) {}
   virtual ~BaseSeries() {}          // so deletes work
   String FN() const { return FileName; }
   String T() const { return Title; }
   virtual String Style() const = 0;
   virtual BaseSeries* Clone() const = 0;  // pointer to a copy of this class
   bool HasValues() const { return FileName != ""; }
};

/// Plot a series as a line.
class LineSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
public:
   LineSeries(const String& title, const Matrix& data, const int colour)
      : BaseSeries(title, data), Colour(ToString(colour)) {}
   LineSeries(const String& title, const Matrix& data, const String& colour)
      : BaseSeries(title, data), Colour(colour) {}
   LineSeries(const LineSeries& bs) : BaseSeries(bs), Colour(bs.Colour) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// Plot a series as impulses (vertical lines).
class ImpulseSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
public:
   ImpulseSeries(const String& title, const Matrix& data, const int colour)
      : BaseSeries(title, data), Colour(ToString(colour)) {}
   ImpulseSeries(const String& title, const Matrix& data, const String& colour)
      : BaseSeries(title, data), Colour(colour) {}
   ImpulseSeries(const ImpulseSeries& bs) : BaseSeries(bs), Colour(bs.Colour) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// Plot a series as steps.
class StepSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
public:
   StepSeries(const String& title, const Matrix& data, const int colour)
      : BaseSeries(title, data), Colour(ToString(colour)) {}
   StepSeries(const String& title, const Matrix& data, const String& colour)
      : BaseSeries(title, data), Colour(colour) {}
   StepSeries(const StepSeries& bs) : BaseSeries(bs), Colour(bs.Colour) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// Plot a series as boxes.
class BoxSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
public:
   BoxSeries(const String& title, const Matrix& data, const int colour)
      : BaseSeries(title, data), Colour(ToString(colour)) {}
   BoxSeries(const String& title, const Matrix& data, const String& colour)
      : BaseSeries(title, data), Colour(colour) {}
   BoxSeries(const BoxSeries& bs) : BaseSeries(bs), Colour(bs.Colour) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};


/// Plot a series as dots.
class DotSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
public:
   DotSeries(const String& title, const Matrix& data, const int colour)
      : BaseSeries(title, data), Colour(ToString(colour)) {}
   DotSeries(const String& title, const Matrix& data, const String& colour)
      : BaseSeries(title, data), Colour(colour) {}
   DotSeries(const DotSeries& bs)
      : BaseSeries(bs), Colour(bs.Colour) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// Plot a series as points.
class PointSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
   String Shape;               ///< Shape of a point.
public:
   PointSeries(const String& title, const Matrix& data,
      const int colour, const int shape)
      : BaseSeries(title, data), Colour(ToString(colour)),
         Shape(ToString(shape)) {}
   PointSeries(const String& title, const Matrix& data,
      const String& colour, const String& shape)
      : BaseSeries(title, data), Colour(colour), Shape(shape) {}
   PointSeries(const PointSeries& bs)
      : BaseSeries(bs), Colour(bs.Colour), Shape(bs.Shape) {}
   String Style() const;
   BaseSeries* Clone() const;        // pointer to a copy of this class
};

/// Plot a series as lines+points.
class LinePointSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
   String Shape;               ///< Shape of a point.
public:
   LinePointSeries(const String& title, const Matrix& data,
      const int colour, const int shape)
      : BaseSeries(title, data), Colour(ToString(colour)),
         Shape(ToString(shape)) {}
   LinePointSeries(const String& title, const Matrix& data,
      const String& colour, const String& shape)
      : BaseSeries(title, data), Colour(colour), Shape(shape) {}
   LinePointSeries(const LinePointSeries& bs)
      : BaseSeries(bs), Colour(bs.Colour), Shape(bs.Shape) {}
   String Style() const;
   BaseSeries* Clone() const;        // pointer to a copy of this class
};

/// Plot a series as error bars in Y direction.
class YErrorBarSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
   String Shape;               ///< Shape of a point.
public:
   YErrorBarSeries(const String& title, const Matrix& data,
      const int colour, const int shape)
      : BaseSeries(title, data), Colour(ToString(colour)),
         Shape(ToString(shape)) {}
   YErrorBarSeries(const String& title, const Matrix& data,
      const String& colour, const String& shape)
      : BaseSeries(title, data), Colour(colour), Shape(shape) {}
   YErrorBarSeries(const YErrorBarSeries& bs)
      : BaseSeries(bs), Colour(bs.Colour), Shape(bs.Shape) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// Plot a series as error bars in X direction.
class XErrorBarSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
   String Shape;               ///< Shape of a point.
public:
   XErrorBarSeries(const String& title, const Matrix& data,
      const int colour, const int shape)
      : BaseSeries(title, data), Colour(ToString(colour)),
         Shape(ToString(shape)) {}
   XErrorBarSeries(const String& title, const Matrix& data,
      const String& colour, const String& shape)
      : BaseSeries(title, data), Colour(colour), Shape(shape) {}
   XErrorBarSeries(const XErrorBarSeries& bs)
      : BaseSeries(bs), Colour(bs.Colour), Shape(bs.Shape) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// Plot a series as error bars in X and Y direction.
class XYErrorBarSeries : public BaseSeries
{
   String Colour;              ///< Colour of a graph.
   String Shape;               ///< Shape of a point.
public:
   XYErrorBarSeries(const String& title, const Matrix& data,
      const int colour, const int shape)
      : BaseSeries(title, data), Colour(ToString(colour)),
         Shape(ToString(shape)) {}
   XYErrorBarSeries(const String& title, const Matrix& data,
      const String& colour, const String& shape)
      : BaseSeries(title, data), Colour(colour), Shape(shape) {}
   XYErrorBarSeries(const XYErrorBarSeries& bs)
      : BaseSeries(bs), Colour(bs.Colour), Shape(bs.Shape) {}
   String Style() const;
   BaseSeries* Clone() const;         // pointer to a copy of this class
};

/// A linked list of Base_Series.
class LL_Series
{
   friend class Plot2;
   friend class Plot3;
   friend class MultiPlot;
   LL_Series* Next;             ///< points to next LL_Series
   BaseSeries* Series;          ///< points to a Series
   LL_Series(LL_Series* next, const BaseSeries& bs);
   ~LL_Series();                ///< delete this and subsequent LL_Series
};

/// Base of graph objects.
class GraphObject
{
   virtual void PlotIt(Plot2&) const = 0;
protected:
   Matrix Location;
   friend class Plot2;
   friend class Plot3;
   friend class MultiPlot;
public:
   virtual ~GraphObject() {}
};

/// Line object.
class Line : public GraphObject
{
   void PlotIt(Plot2&) const;
public:
   Line(Real x1, Real y1, Real x2, Real y2);
   Line(Real x1, Real y1, Real z1, Real x2, Real y2, Real z2);
   friend class Plot2;
   friend class Plot3;
   friend class MultiPlot;
};

/// Arrow object.
class Arrow : public GraphObject
{
   void PlotIt(Plot2&) const;
public:
   Arrow(Real x1, Real y1, Real x2, Real y2);
   Arrow(Real x1, Real y1, Real z1, Real x2, Real y2, Real z2);
   friend class Plot2;
   friend class Plot3;
   friend class MultiPlot;
};

/// Label (text) object.
class Label : public GraphObject
{
public:
   enum Justification { left, right, centre, center };
private:
   void PlotIt(Plot2&) const;
   String TheLabel;
   Justification justify;
public:
   Label(const String& label, Real x, Real y, Justification j);
   Label(const String& label, Real x, Real y, Real z, Justification j);
   friend class Plot2;
   friend class Plot3;
   friend class MultiPlot;
};



/// Axis of a graph.
class Axis
{
   String Title;                   ///< label on axis
   bool IsLog;                     ///< true for log transform
   bool Wrange;                    ///< true if range set
   bool IsSet;                     ///< true if values are set
   bool ZeroAxis;                  ///< true if we want a zero axis
   Real Max;                       ///< range maximum
   Real Min;                       ///< range minimum
   Axis() : IsSet(false) {}        // values not set
public:
   Axis(const String& title) : Title(title), IsLog(false), Wrange(false),
      IsSet(true), ZeroAxis(false) {}
   void SetLog(bool il = true) { IsLog = il; }
                                   ///< set or reset log transform
   void SetRange(Real min, Real max) { Min=min; Max=max; Wrange = true; }
                                   ///< set range of Axis
   void SetZeroAxis(bool za = true) { ZeroAxis = za; }
                                   ///< set or reset axis through zero
   friend class Plot2;
   friend class Plot3;
   friend class MultiPlot;
   // use default operator=
};

/// Two dimensional plot.
class Plot2
{
   friend class MultiPlot;
protected:
   String Title;                  ///< Title of plot
   String Commands;               ///< String of commands sent to gnuplot
   String Functions;              ///< String of functions sent to gnuplot
   LL_Series* LLS;                ///< Linked list of graphs
   Axis X_Axis;                   ///< X Axis
   Axis Y_Axis;                   ///< Y Axis
   bool WK;                       ///< true if we want the key (legend)
   double KeySpacing;             ///< vertical spacing of key
   bool WI;                       ///< true for reading points interactively
public:
   Plot2(const String& title)
      : Title(title), Commands(""), Functions(""), LLS(0), WK(true),
         KeySpacing(1.0), WI(false) {}
   Plot2& operator<< (const BaseSeries&);
                                  ///< Add a graph to a plot.
   Plot2& operator<< (const GraphObject& go)
      { go.PlotIt(*this); return *this; }
                                  ///< Add a simple object to a plot.
   void AddCommand(const String& command)
      { Commands += command; Commands += "\n"; }
                                  ///< Add a command to command string.
   void PartCommand(const String& command) { Commands += command; }
                                  ///< Add a string to cammand string (no CR).
   void Set(const String& command)
      { Commands += "set "; Commands += command; Commands += "\n"; }
                                  ///< Add a set command to command string.
   void Unset(const String& command)
      { Commands += "unset "; Commands += command; Commands += "\n"; }
                                  ///< Add an unset command to command string.
   void AddFunction(const String& function);
   void Set_X_Axis(const Axis& xa) { X_Axis = xa; }
                                  ///< Copy X axis to plot.
   void Set_Y_Axis(const Axis& ya) { Y_Axis = ya; }
                                  ///< Copy Y axis to plot.
   virtual void DoPlot();
   void WantKey(bool wk = true) { WK = wk; }
                                  ///< Want or don't want key (legend)
   void WantKey(double ks, bool wk = true) { KeySpacing = ks; WK = wk; }
                                  ///< Set key (legend) spacing
   void WantKey(int ks, bool wk = true) { KeySpacing = ks; WK = wk; }
                                  ///< Set key (legend) spacing
   void Interact(bool wi = true) { WI = wi; }
                                  ///< For reading points interactively
   virtual ~Plot2();                      // destroy the linked list
};

/// Three dimensional plot.
class Plot3 : public Plot2
{
   friend class MultiPlot;
protected:
   Axis Z_Axis;                   ///< Z Axis
public:
   Plot3(const String& title) : Plot2(title) {}
   void Set_Z_Axis(const Axis& za) { Z_Axis = za; }
                                  ///< Copy Z axis to plot.
   void DoPlot();
};

/// Plot with subplots.
class MultiPlot
{
   String Commands;
   String Title;
public:
   MultiPlot();
   MultiPlot(const String& title);
   ~MultiPlot() {}
   void Size(Real x, Real y);
   void Origin(Real x, Real y);
   void PointSize(Real ps);
   MultiPlot& operator<< (const Plot2& plot);
   MultiPlot& operator<< (const Plot3& plot);
   void Set(const String& command)
      { Commands += "set "; Commands += command; Commands += "\n"; }
   void Unset(const String& command)
      { Commands += "unset "; Commands += command; Commands += "\n"; }
   void DoPlot();
};

#ifdef use_namespace
}
#endif



#endif


// body file: gnuplot.cpp


///@}



