// genmake.cpp
// version of 18 December, 2007

// Generate a make file

#define WANT_STREAM
#define WANT_MATH
#define WANT_TIME
#define WANT_FSTREAM

#include "include.h"
#include "str.h"
#include "gstring.h"
#include "commline.h"


// *** Classes to customise output for particular compilers and make files ***

class CompilerProperties
{
   String code;
public:
   CompilerProperties(const char* c) : code(c) {}
   virtual ~CompilerProperties() {}
   void Preamble() const;
   virtual void LinkStatement(const String& Name, const String& FN) const = 0;
   virtual void LinkStatement(const String& Name, const StringList& Libraries)
      const = 0;
   virtual void LibStatement(const String& Name) const = 0;
   virtual const char* ObjectType() const = 0;
   virtual int ObjectSize() const = 0;
   virtual const char* ExeType() const = 0;
   virtual int ExeSize() const = 0;
   virtual char FS() const = 0;
   virtual bool DoLibrary() const { return true; }
   virtual bool WantPlusVersion() const { return false; }
protected:
   void LibraryList(const char* prefix, const StringList& Libraries,
      const char* suffix) const;
};

class WindowsCompiler : public CompilerProperties
{
protected:
   WindowsCompiler(const char* c) : CompilerProperties(c) {}
public:
   const char* ObjectType() const { return ".obj"; }
   int ObjectSize() const { return 4; }
   const char* ExeType() const { return ".exe"; }
   int ExeSize() const { return 4; }
   char FS() const  { return '\\'; }
};

class UnixCompiler : public CompilerProperties
{
protected:
   UnixCompiler(const char* c) : CompilerProperties(c) {}
public:
   const char* ObjectType() const { return ".o"; }
   int ObjectSize() const { return 2; }
   const char* ExeType() const { return ""; }
   int ExeSize() const { return 0; }
   char FS() const  { return '/'; }
};


class GnuCompiler : public UnixCompiler
{
public:
   GnuCompiler(const char* c) : UnixCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const;
   void LibStatement(const String& Name) const;
};

class GnuCompilerDynamic : public GnuCompiler
{
public:
   GnuCompilerDynamic(const char* c) : GnuCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const
      { GnuCompiler::LinkStatement(Name, FN); }
   void LinkStatement(const String& Name, const StringList& Libraries) const;
   void LibStatement(const String& Name) const;
};

class Borland55Compiler : public WindowsCompiler
{
public:
   Borland55Compiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const;
   void LibStatement(const String& Name) const;
};

class Borland50dCompiler : public WindowsCompiler
{
public:
   Borland50dCompiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const {}
   void LibStatement(const String& Name) const {}
   bool DoLibrary() const { return false; }  // don't do libraries
};

class Borland50Compiler : public WindowsCompiler
{
public:
   Borland50Compiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const {}
   void LibStatement(const String& Name) const {}
   bool DoLibrary() const { return false; }  // don't do libraries
};

class Borland31Compiler : public WindowsCompiler
{
public:
   Borland31Compiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const {}
   void LibStatement(const String& Name) const {}
   bool DoLibrary() const { return false; }  // don't do libraries
};

class Microsoft6Compiler : public WindowsCompiler
{
public:
   Microsoft6Compiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const;
   void LibStatement(const String& Name) const;
};

class Watcom10Compiler : public WindowsCompiler
{
public:
   Watcom10Compiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const {}
   void LibStatement(const String& Name) const {}
   bool DoLibrary() const { return false; }  // don't do libraries
};

class OpenWatcomCompiler : public WindowsCompiler
{
public:
   OpenWatcomCompiler(const char* c) : WindowsCompiler(c) {}
   void LinkStatement(const String& Name, const String& FN) const;
   void LinkStatement(const String& Name, const StringList& Libraries) const;
   void LibStatement(const String& Name) const;
   virtual bool WantPlusVersion() const { return true; } // lib list with +s
};


// elapsed time class

class time_lapse
{
   double start_time;
public:
   time_lapse();
   ~time_lapse();
};


// ************* function headers ********************

// Does a string list, LS, contain a particular string S

bool Contains(const StringList& LS, const String& S);

// search a source file for header and body file names - then
// search these files and so on. Return list of header and body
// file names

bool SearchHB(const String& SourceFile, StringList& Headers,
   StringList& Bodies);

// search a source file for header file names - then search
// these files and so on. Return list of header file names

bool SearchH(const String& SourceFile, StringList& Headers);

// find header files in .lfl file then apply SearchHB to these
// NewHeaders are the names just in the SourceFile
// return false if source files missing

bool SearchL(const String& SourceFile, StringList& Headers,
   StringList& NewHeaders, StringList& Bodies);

// find list of libraries required and purged list of headers
// return true if any libraries found
// no output if libraries not found

bool IdentifyLibraries(
   const StringList& Headers,       // input list of headers
   const StringList& Libraries,     // library files available
   const StringList& LibraryFiles,  // all headers whose body is in library
   const StringList& LibraryFiles1, // headers which trigger inclusion
   StringList& LibrariesWanted,     // return list of libraries required
   StringList& PurgedHeaders);      // return headers which are not matched

// find bodies referenced in a list of header files - no recursion

void GetBodies(const StringList& Headers, StringList& Bodies);

// output n spaces

void Spaces(int n) { while (n-- > 0) cout << ' '; }


// ************* main program section ********************

typedef StringList::iterator SLI;
typedef StringList::const_iterator SLCI;

int main(int argc, char** argv)
{
   Try
   {
      time_lapse tl;      // measure program run time
      CommandLine CL(argc, argv);
      int N = CL.NumberOfArgs();

      if (N <= 0) { cerr << "No targets" << endl; exit(1); }

      bool WantCompare = CL.HasOption("!");    // compare output with text file

      // Get compiler type and construct corresponding class

      CompilerProperties* CP;

      if (CL.HasOptionCI("G") && CL.HasOptionCI("D"))
         CP = new GnuCompilerDynamic("pre_g.txt");

      else if (CL.HasOptionCI("G") && CL.HasOptionCI("2"))
         CP = new GnuCompiler("pre_g2.txt");

      else if (CL.HasOptionCI("G"))
         CP = new GnuCompiler("pre_g.txt");

      else if (CL.HasOptionCI("C"))
         CP = new GnuCompiler("pre_c.txt");

      else if (CL.HasOptionCI("B") && CL.HasOptionCI("D")  && CL.HasOption("0"))
         CP = new Borland50dCompiler("pre_b0d.txt");

      else if (CL.HasOptionCI("B") && CL.HasOption("0"))
         CP = new Borland50Compiler("pre_b0.txt");

      else if (CL.HasOptionCI("B") && CL.HasOption("8"))
         CP = new Borland55Compiler("pre_b8.txt");

      else if (CL.HasOptionCI("B") && CL.HasOption("6"))
         CP = new Borland55Compiler("pre_b6.txt");

      else if (CL.HasOptionCI("B") && CL.HasOption("5"))
         CP = new Borland55Compiler("pre_b5.txt");

      else if (CL.HasOptionCI("B") && CL.HasOption("3")&& CL.HasOption("1"))
         CP = new Borland31Compiler("pre_b31.txt");

      else if (CL.HasOptionCI("M") && CL.HasOption("8"))
         CP = new Microsoft6Compiler("pre_m8.txt");

      else if (CL.HasOptionCI("M") && CL.HasOption("6"))
         CP = new Microsoft6Compiler("pre_m6.txt");

      else if (CL.HasOptionCI("M") && CL.HasOption("5"))
         CP = new Microsoft6Compiler("pre_m5.txt");

      else if (CL.HasOptionCI("I") && CL.HasOptionCI("L") && CL.HasOptionCI("D")
         && CL.HasOption("5"))
         CP = new GnuCompilerDynamic("pre_il5.txt");

      else if (CL.HasOptionCI("I") && CL.HasOptionCI("L") && CL.HasOptionCI("D")
         && CL.HasOption("8"))
         CP = new GnuCompilerDynamic("pre_il8.txt");

      else if (CL.HasOptionCI("I") && CL.HasOptionCI("L") && CL.HasOption("5"))
         CP = new GnuCompiler("pre_il5.txt");

      else if (CL.HasOptionCI("I") && CL.HasOptionCI("L") && CL.HasOption("8"))
         CP = new GnuCompiler("pre_il8.txt");

      else if (CL.HasOptionCI("I") && CL.HasOption("5"))
         CP = new Microsoft6Compiler("pre_i5.txt");

      else if (CL.HasOptionCI("I") && CL.HasOption("8"))
         CP = new Microsoft6Compiler("pre_i8.txt");

      else if (CL.HasOptionCI("I") && CL.HasOption("1") && CL.HasOption("0"))
         CP = new Microsoft6Compiler("pre_i10.txt");

      else if (CL.HasOptionCI("O") && CL.HasOptionCI("W"))
         CP = new OpenWatcomCompiler("pre_ow.txt");

      else if (CL.HasOptionCI("W") && CL.HasOptionCI("D") && CL.HasOption("0"))
         CP = new Watcom10Compiler("pre_w0d.txt");

      else if (CL.HasOptionCI("W") && CL.HasOption("0"))
         CP = new Watcom10Compiler("pre_w0.txt");

      else { cerr << "Unrecognised compiler option" << endl; exit(1); }

      // print preamble

      CP->Preamble();

      // get targets; separate out libraries
      // beginning with @ means argument file
      // beginning with I= or L= means include or library directory

      StringList Targets;
      StringList Libraries;

      StringList LibraryDirectories;
      StringList IncludeDirectories;

      cout << "everything:    \t";

      for (int ni = 1; ni <= N; ++ni)
      {
         String s = CL.GetArg(ni);
         if (s[0] == '@')
         {
            ifstream is(String(s,1).c_str(),ios::in);
            if (!is) { cerr << "No argument file " << s << endl; }
            else
            {
               StringList Args; is >> Args;
               if (Args.size() == 0 ||
                  (Args.size() == 1 && Args.begin()->size() == 0))
                  { cerr << "File " << s << " has no data" << endl; }

               for (SLI ai = Args.begin(); ai != Args.end(); ++ai)
               {
                  if (ai->size() > 2 && ai->substr(0,2) == "I=")
                     IncludeDirectories.push_back(ai->substr(2));
                  else if (ai->size() > 2 && ai->substr(0,2) == "L=")
                     LibraryDirectories.push_back(ai->substr(2));
                  else
                  {
                     uint sf = ai->find(".lfl");
                     if (sf != String::npos) Libraries.push_back(ai->erase(sf));
                     else if (ai->find_first_not_of(' ') != String::npos)
                     {
                        Targets.push_back(*ai);
                        String sx = *ai;
                        sf = sx.find(".c");     // remove extension
                        if (sf != String::npos) sx.erase(sf);
                        if (WantCompare) cout << sx << ".txx ";
                        else cout << sx << CP->ExeType() << " ";
                     }
                  }
               }
            }
         }
         else if (s.size() > 2 && s.substr(0,2) == "I=")
            IncludeDirectories.push_back(s.substr(2));
         else if (s.size() > 2 && s.substr(0,2) == "L=")
            LibraryDirectories.push_back(s.substr(2));
         else
         {
            uint sf = s.find(".lfl");
            if (sf != String::npos) Libraries.push_back(s.erase(sf));
            else
            {
               Targets.push_back( s );
               String sx = s;
               sf = sx.find(".c");     // remove extension
               if (sf != String::npos) sx.erase(sf);
               if (WantCompare) cout << sx << ".txx ";
               else cout << sx << CP->ExeType() << " ";
            }
         }
      }
      cout << endl << endl;


      // find the bodies corresponding to each library
      // skip DoLibrary returns false

      StringList Bodies;                // to hold names of .cpp files
      StringList LibraryFiles;
      StringList LibraryFiles1;

      if (CP->DoLibrary())
      {
         for (SLI i = Libraries.begin(); i != Libraries.end(); ++i)
         {
            SLI j; bool found = true;
            StringList NB;              // Bodies corresponding to this target
            StringList NH;              // Headers including implied headers
            StringList NH1;             // Headers excluding implied headers
            String Main = *i+".lfl";
            if (!SearchL(Main, NH, NH1, NB))
            {
               cerr << "Assume library " << *i << " up-to-date"
                  << endl << endl;
               cout << "# Assume library " << *i << " up-to-date"
                  << endl << endl;
               found = false;
            }
            cout << *i << "_lobj =";
            for (j = NB.begin(); j != NB.end(); ++j)
            {
               String object = *j;
               uint f = object.find(".c");
               if (f == String::npos)
                  { cerr << "Invalid body type " << object << endl; exit(1); }
               cout << " " << object.erase(f) << CP->ObjectType();
               if (!Contains(Bodies,*j)) Bodies.push_back(*j);
            }
            cout << endl << endl;
            
            if (CP->WantPlusVersion())
            {
               cout << *i << "_pobj =";
               for (j = NB.begin(); j != NB.end(); ++j)
               {
                  String object = *j;
                  uint f = object.find(".c");
                  if (f == String::npos)
                     { cerr << "Invalid body type " << object << endl; exit(1); }
                  cout << " +" << object.erase(f) << CP->ObjectType();
               }
               cout << endl << endl;
            }

            String LF = " ";            // names of files making up library
            for (j = NH.begin(); j != NH.end(); ++j) LF += (*j + " ");
            LibraryFiles.push_back(LF);
            LF = " ";
            for (j = NH1.begin(); j != NH1.end(); ++j) LF += (*j + " ");
            LibraryFiles1.push_back(LF);
            if (found) CP->LibStatement(*i);
         }
      }

      // find the bodies corresponding to each target

      SLI i;
      for (i = Targets.begin(); i != Targets.end(); ++i)
      {
         StringList NH;                 // Headers corresponding to this target
         StringList NB;                 // Bodies corresponding to this target
         StringList L;                  // Libraries
         StringList LW;                 // Libraries wanted
         StringList PH;                 // Purged headers
         String Main = *i;              // target file name
         uint f = Main.find(".c");
         if (f == String::npos)         // no extension
            Main += ".cpp";             // standard extension
         else i->erase(f);              // delete extension
         NB.push_back(Main);
         SearchHB(Main, NH, NB);

         // find what libraries we need and what headers are left
         bool WantLibraries = IdentifyLibraries(NH, Libraries,
            LibraryFiles, LibraryFiles1, LW, PH);

         if (WantLibraries)
         {
            NB.CleanUp();
            NB.push_back(Main);
            GetBodies(PH, NB);
            cout << *i << "_obj =";
            for (SLI j = NB.begin(); j != NB.end(); ++j)
            {
               String object = *j;
               uint f = object.find(".c");
               if (f == String::npos)
                  { cerr << "Invalid body type " << object << endl; exit(1); }
               object.erase(f);
               cout << " " << object << CP->ObjectType();
               if (!Contains(Bodies,*j)) Bodies.push_back(*j);
            }
            cout << endl << endl;
            CP->LinkStatement(*i, LW);
         }
         else
         {
            String FN;   // list of file names; may be required by LinkStatement
            cout << *i << "_obj =";
            for (SLI j = NB.begin(); j != NB.end(); ++j)
            {
               String object = *j;
               uint f = object.find(".c");
               if (f == String::npos)
                  { cerr << "Invalid body type " << object << endl; exit(1); }
               object.erase(f);
               cout << " " << object << CP->ObjectType();
               if (!Contains(Bodies,*j)) Bodies.push_back(*j);
               FN += (" " + object + CP->ObjectType());
            }
            cout << endl << endl;
            CP->LinkStatement(*i, FN);
         }
      }


      // find the headers corresponding to each body

      for (i = Bodies.begin(); i != Bodies.end(); ++i)
      {
         String object = *i;
         uint f = object.find(".c");
         if (f == String::npos)
           { cerr << "Invalid body type " << object << endl; exit(1); }
         object.erase(f);
         StringList Headers;
         if (SearchH(*i, Headers))
         {
            cout << object << CP->ObjectType() << ":";
            Spaces(14 - object.size() - CP->ObjectSize());
            cout << "\t" << *i;
            for (SLI j = Headers.begin(); j != Headers.end(); ++j)
               cout << " " << *j;
            cout << endl;
         }
         cout << endl;
      }

      // lines for checking output of program

      for (i = Targets.begin(); i != Targets.end(); ++i)
      {
         cout << *i << ".txx:";
         Spaces(10 - i->size());
         cout << "\t" << *i << CP->ExeType() << endl;
         cout << "\t\t$(PRE)" << *i << " > " << *i << ".txx" << endl;
         cout << "\t\t$(DIFF) " << *i << ".txt " << *i << ".txx" << endl;
         cout << endl;
      }
   }
   CatchAll
   {
      cerr << endl;
      cerr << "Program fails - exception thrown" << endl;
      cerr << Exception::what() << endl;
   }

   return 0;

}

// *************** function bodies ******************

// Does a string list, LS, contain a particular string S

bool Contains(const StringList& LS, const String& S)
{
   for (SLCI i = LS.begin(); i != LS.end(); ++i) { if (*i == S) return true; }
   return false;
}

// search a source file for header and body file names - then
// search these files and so on. Return list of header and body
// file names

bool SearchHB(const String& SourceFile, StringList& Headers, StringList& Bodies)
{
   StringList NewHeaders;
   StringList NewBodies;

   {
      ifstream is(SourceFile.c_str(),ios::in);
      if (!(is))
      {
         cerr << "File " << SourceFile << " not found" << endl;
         return false;
      }

      StringList Source;                // to hold source of a file
      is >> Source;                     // get the source
      if (Source.size() == 0 ||
         (Source.size() == 1 && Source.begin()->size() == 0))
      {
         cerr << "File " << SourceFile << " has no data" << endl;
         return false;
      }

      for (SLI i = Source.begin(); i != Source.end(); ++i)
      {
         AnyString AS;
         if (( OWS+"#include" > OWS+"\""+AS+"\""+DOTS ).Matches(*i)
           && !Contains(Headers,AS.Value()))
         {
            Headers.push_back(AS.Value());
            NewHeaders.push_back(AS.Value());
         }
         if (( OWS+"// body file:" > OWS+AS+OWS ).Matches(*i)
           && !Contains(Bodies,AS.Value()))
         {
            Bodies.push_back(AS.Value());
            NewBodies.push_back(AS.Value());
         }
      }
   }

   SLI i; bool found = true;
   for (i = NewHeaders.begin(); i != NewHeaders.end(); ++i)
      found = SearchHB(*i, Headers, Bodies) && found;
   for (i = NewBodies.begin(); i != NewBodies.end(); ++i)
      found = SearchHB(*i, Headers, Bodies) && found;

   return found;
}

// search a source file for header file names - then search
// these files and so on. Return list of header file names

bool SearchH(const String& SourceFile, StringList& Headers)
{
   StringList NewHeaders;
   StringList NewBodies;

   {
      ifstream is(SourceFile.c_str(),ios::in);

      if (is)
      {
         StringList Source;                // to hold source of a file
         is >> Source;                     // get the source
         if (Source.size() == 0 ||
            (Source.size() == 1 && Source.begin()->size() == 0))
         {
            cerr << "File " << SourceFile << " has no data" << endl;
            return false;
         }

         for (SLI i = Source.begin(); i != Source.end(); ++i)
         {
            AnyString AS;
            if (( OWS+"#include" > OWS+"\""+AS+"\""+DOTS ).Matches(*i)
              && !Contains(Headers,AS.Value()))
            {
               Headers.push_back(AS.Value());
               NewHeaders.push_back(AS.Value());
            }
         }
      }
      else
      {
         cerr << "File " << SourceFile << " not found" << endl;
         return false;
      }

   }

   bool found = true;
   for (SLI i = NewHeaders.begin(); i != NewHeaders.end(); ++i)
      found = SearchH(*i, Headers) && found;

   return found;
}

// find header files in .lfl file then apply SearchHB to these
// NewHeaders are the names just in the SourceFile
// return false if source files missing

bool SearchL(const String& SourceFile, StringList& Headers,
   StringList& NewHeaders, StringList& Bodies)
{

   {
      ifstream is(SourceFile.c_str(),ios::in);
      if (!(is))
      {
         cerr << "File " << SourceFile << " not found" << endl;
         return false;
      }

      StringList Source;                // to hold source of a file
      is >> Source;                     // get the source
      if (Source.size() == 0 ||
         (Source.size() == 1 && Source.begin()->size() == 0))
      {
         cerr << "File " << SourceFile << " has no data" << endl;
         return false;
      }

      for (SLI i = Source.begin(); i != Source.end(); ++i)
      {
         if (i->find(".h") != String::npos && !Contains(NewHeaders,*i))
            { Headers.push_back(*i); NewHeaders.push_back(*i); }
      }
   }

   bool found = true;
   for (SLI i = NewHeaders.begin(); i != NewHeaders.end(); ++i)
      found = SearchHB(*i, Headers, Bodies) && found;

   return found;
}

// find list of libraries required and purged list of headers
// return true if any libraries found
// no output if libraries not found

bool IdentifyLibraries(const StringList& Headers,
   const StringList& Libraries,
   const StringList& LibraryFiles,
   const StringList& LibraryFiles1,
   StringList& LibrariesWanted,
   StringList& PurgedHeaders)
{
   SLCI j;
   String LFA;                                // Header files available
   bool found = false;
   // see what libraries we want
   for (j = Headers.begin(); j != Headers.end(); ++j)
   {
      for ( SLCI L = Libraries.begin(), LF = LibraryFiles.begin(),
         LF1 = LibraryFiles1.begin(); L != Libraries.end();
         ++L, ++LF, ++LF1)
      {
         if (LF1->find(' ' + *j + ' ') != String::npos)
         {
            if (!Contains(LibrariesWanted, *L))
               { LibrariesWanted.push_back(*L); LFA += *LF; }
            found = true; break;
         }
      }
   }
   if (!found)                                 // no libraries
   {
      for (SLCI j = Headers.begin(); j != Headers.end(); ++j)
         PurgedHeaders.push_back(*j);
      return false;
   }

   // see what headers are left
   for (j = Headers.begin(); j != Headers.end(); ++j)
   {
      if (LFA.find(' ' + *j + ' ') == String::npos)
         PurgedHeaders.push_back(*j);
   }

   return true;

}

// find bodies referenced in a list of header files - no recursion

void GetBodies(const StringList& Headers, StringList& Bodies)
{
   for (SLCI i = Headers.begin(); i != Headers.end(); ++i)
   {
      ifstream is(i->c_str(),ios::in);
      if (!(is))
         { cerr << "File " << *i << " not found" << endl; break; }

      StringList Source;                // to hold source of a file
      is >> Source;                     // get the source
      if (Source.size() == 0 ||
         (Source.size() == 1 && Source.begin()->size() == 0))
      {
         cerr << "File " << *i << " has no data" << endl;
         break;
      }

      for (SLI j = Source.begin(); j != Source.end(); ++j)
      {
         AnyString AS;
         if (( OWS+"// body file:" > OWS+AS+OWS ).Matches(*j)
            && !Contains(Bodies,AS.Value()))
            Bodies.push_back(AS.Value());
      }
   }
}


// **************** Compiler properties *****************

void CompilerProperties::Preamble() const
{
   StringList SL;
   ifstream is(code.c_str(),ios::in);
   if (!(is)) { cerr << "File " << code << " not found" << endl; exit(1); }
   is >> SL;
   if (SL.size() == 0 || (SL.size() == 1 && SL.begin()->size() == 0))
       { cerr << "File " << code << " has no data" << endl; exit(1); }
   cout << SL;
}

// print a list of libraries

void CompilerProperties::LibraryList(const char* prefix,
   const StringList& Libraries, const char* suffix) const
{
   for (SLCI j = Libraries.begin(); j != Libraries.end(); ++j)
      cout << prefix << *j << suffix;
}


// *************** Gnu G++ compiler **********************

void GnuCompiler::LinkStatement(const String& Name, const String&) const
{
   cout << Name << ":";
   Spaces(14 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "\t\t$(CXX) -o $@ $(" << Name << "_obj) -L. -lm" << endl;
   cout << endl;
}

void GnuCompiler::LinkStatement(const String& Name,
   const StringList& Libraries) const
{
   cout << Name << ":";
   Spaces(14 - Name.size());
   cout << "\t$(" << Name << "_obj)";
   LibraryList(" lib", Libraries, ".a");
   cout << endl;
   cout << "\t\t$(CXX) -o $@ $(" << Name << "_obj) -L.";
   LibraryList(" -l", Libraries, "");
   cout << " -lm" << endl;
   cout << endl;
}

void GnuCompiler::LibStatement(const String& Name) const
{
   cout << "lib" << Name << ".a:";
   Spaces(9 - Name.size());
   cout << "\t$(" << Name << "_lobj)" << endl;
   cout << "\t\t$(AR) -cr $@ $(" << Name << "_lobj)" << endl;
   cout << "\t\tranlib $@" << endl;
   cout << endl;
}

// ********** Gnu G++ compiler, Dynamic version ***********

void GnuCompilerDynamic::LinkStatement(const String& Name,
   const StringList& Libraries) const
{
   cout << Name << ":";
   Spaces(14 - Name.size());
   cout << "\t$(" << Name << "_obj)";
   LibraryList(" lib", Libraries, ".so");
   cout << endl;
   cout << "\t\t$(CXX) -o $@ $(" << Name << "_obj) -L.";
   LibraryList(" -l", Libraries, "");
   cout << " -lm" << endl;
   cout << endl;
}

void GnuCompilerDynamic::LibStatement(const String& Name) const
{
   String libname = "lib" + Name + ".so";
   cout << libname << ':';
   Spaces(8 - Name.size());
   cout << "\t$(" << Name << "_lobj)" << endl;
   cout << "\t\t$(CXX) -shared -W1,-soname,";
   cout << libname << ".$(MAJOR) -o ";
   cout << libname << ".$(MAJOR).$(MINOR) $(" << Name << "_lobj) -lc" << endl;
   cout << "\t\trm -f " << libname << ".$(MAJOR)" << endl;
   cout << "\t\trm -f " << libname << endl;
   cout << "\t\tln -s " << libname << ".$(MAJOR).$(MINOR) "
      << libname << ".$(MAJOR)" << endl;
   cout << "\t\tln -s " << libname << ".$(MAJOR) " << libname << endl;
   cout << endl;
}

// ****************** Borland 5.0 **********************

void Borland50dCompiler::LinkStatement(const String& Name, const String& FN)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "   $(TLINK) /x/L$(LIBPATH) -c -Tde @&&|" << endl;
   cout << "c0l.obj";
   for (unsigned int i = 0; i < FN.size(); ++i)
      { if (FN[i]==' ') cout << '+' << endl; else cout << FN[i]; }
   cout << endl << "$<,$*" << endl;
   cout << "bidsl.lib+emu.lib+mathl.lib+cl.lib" << endl;
   cout << endl;
   cout << "|" << endl << endl;
}

void Borland50Compiler::LinkStatement(const String& Name, const String& FN)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "   $(TLINK) /x/L$(LIBPATH) -Tpe -ap -c @&&|" << endl;
   cout << "c0x32.obj";
   // Borland requires separate lines
   for (unsigned int i = 0; i < FN.size(); ++i)
      { if (FN[i]==' ') cout << '+' << endl; else cout << FN[i]; }
   cout << endl << "$<,$*" << endl;
   cout << "bidsf.lib+" << endl;
   cout << "import32.lib+" << endl;
   cout << "cw32.lib" << endl;
   cout << endl;
   cout << "|" << endl << endl;
}

// ****************** Borland 5.5 **********************

void Borland55Compiler::LinkStatement(const String& Name, const String&)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "   $(TLINK) /x/L$(LIBPATH)/Gn -Tpe -ap -c @&&|" << endl;
   cout << "c0x32.obj $(" << Name << "_obj),$@,,import32.lib cw32.lib" << endl;
   cout << "|" << endl << endl;
}

void Borland55Compiler::LinkStatement(const String& Name,
   const StringList& Libraries) const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)";
   LibraryList(" ", Libraries, ".lib");
   cout << endl;
   cout << "   $(TLINK) /x/L$(LIBPATH)/Gn -Tpe -ap -c @&&|" << endl;
   cout << "c0x32.obj $(" << Name << "_obj),$@,,";
   LibraryList(" ", Libraries, ".lib");
   cout << " import32.lib cw32.lib" << endl;
   cout << "|" << endl << endl;
}

void Borland55Compiler::LibStatement(const String& Name) const
{
   cout << Name << ".lib:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_lobj)" << endl;
   cout << "   $(TLIB) $@ /P32 /u $(" << Name << "_lobj)" << endl;
   cout << endl;
}

// ****************** Borland 3.1 **************************

void Borland31Compiler::LinkStatement(const String& Name, const String& FN)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "   $(TLINK) /x -c -Tde @&&|" << endl;
   cout << "c0l.obj";
   for (unsigned int i = 0; i < FN.size(); ++i)
      { if (FN[i]==' ') cout << '+' << endl; else cout << FN[i]; }
   cout << endl << "$<,$*" << endl;
   cout << "emu.lib+mathl.lib+cl.lib" << endl;
   cout << endl;
   cout << "|" << endl << endl;
}


// ****************** Microsoft 6 **************************

void Microsoft6Compiler::LinkStatement(const String& Name, const String&)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "\t\tlink -Out:$@ $(conlibs) $(" << Name << "_obj)" << endl;
   cout << endl;
}

void Microsoft6Compiler::LinkStatement(const String& Name,
   const StringList& Libraries) const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)";
   LibraryList(" ", Libraries, ".lib");
   cout << endl;
   cout << "\t\tlink -Out:$@ $(conlibs) $(" << Name << "_obj)";
   LibraryList(" ", Libraries, ".lib");
   cout << endl << endl;
}

void Microsoft6Compiler::LibStatement(const String& Name) const
{
   cout << Name << ".lib:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_lobj)" << endl;
   cout << "\t\tlib -Out:$@ $(" << Name << "_lobj)" << endl;
   cout << endl;
}

// *************** Watcom 10 **********************

void Watcom10Compiler::LinkStatement(const String& Name, const String& FN)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   int k = 0;
   for (unsigned int i = 0; i < FN.size(); ++i)
   {
      if (FN[i]==' ')
      {
         if (k == 1) cout << " > link.tmp" << endl;
         else if (k >= 2) cout << " >> link.tmp" << endl;
         ++k;
         cout << "\t\techo file ";
      }
      else cout << FN[i];
   }
   if (k == 1) cout << " > link.tmp" << endl;
   else if (k >= 2) cout << " >> link.tmp" << endl;
   cout << "\t\techo name " << Name << ".exe >> link.tmp" << endl;
   cout << "\t\techo $(link_option) >> link.tmp" << endl;
   cout << "\t\t$(LINK) @link.tmp" << endl;
   cout << endl;
}

// *************** Open Watcom *********************

void OpenWatcomCompiler::LinkStatement(const String& Name, const String&)
   const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)" << endl;
   cout << "\t\twcl386 -fe=$@ $(" << Name << "_obj)" << endl;
   cout << endl;
}

void OpenWatcomCompiler::LinkStatement(const String& Name,
   const StringList& Libraries) const
{
   cout << Name << ".exe:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_obj)";
   LibraryList(" ", Libraries, ".lib");
   cout << endl;
   cout << "\t\twcl386 -fe=$@ $(" << Name << "_obj)";
   LibraryList(" ", Libraries, ".lib");
   cout << endl << endl;
}

void OpenWatcomCompiler::LibStatement(const String& Name) const
{
   cout << Name << ".lib:";
   Spaces(10 - Name.size());
   cout << "\t$(" << Name << "_lobj)" << endl;
   cout << "\t\twlib -n $@ $(" << Name << "_pobj)" << endl;
   cout << endl;
}







//************** elapsed time class ****************

time_lapse::time_lapse()
{
   start_time = ((double)clock())/(double)CLOCKS_PER_SEC;
}

time_lapse::~time_lapse()
{
   double time = ((double)clock())/(double)CLOCKS_PER_SEC - start_time;
   cerr << "Elapsed (processor) time = " << setw(10) << setprecision(2)
      << time << " seconds" << endl;
   cerr << endl;
}

