//$ cx.h                     complex numbers definition file

// "WHAT THE STANDARD SHOULD HAVE BEEN LIKE"

#ifndef CX_LIB
#define CX_LIB

#include "include.h"
#include "myexcept.h"

#ifdef use_namespace
namespace RBD_COMPLEX { using namespace RBD_COMMON; }
namespace RBD_LIBRARIES { using namespace RBD_COMPLEX; }
namespace RBD_COMPLEX {
using RBD_COMMON::Real;
using RBD_COMMON::long_Real;
#endif



class ConvertFromReal;      // used for stopping conversions by a constructor
class ImaginaryUnit;        // i or j in complex algebra
class CX;                   // complex class
class Imag;                 // pure imaginary numbers
class Polar;                // imaginary in polar coordinates
class Quadrant;             // mod 4 arithmetic

extern Real pi, pi_times_2, pi_over_2, pi_over_4;

Real ipow(Real x, int n);   // if pow with integer n isn't available



class ConvertFromReal
{
   Real CFR;
public:
   ConvertFromReal(Real cfr) : CFR(cfr) {}
   Real Value() const { return CFR; }
};

class ImaginaryUnit
{
public:
   Real operator*(ImaginaryUnit) { return -1.0; }
   Real operator/(ImaginaryUnit) { return 1.0; }
   ImaginaryUnit operator+();
   Imag operator-();
   bool operator==(ImaginaryUnit) { return true; }
   bool operator!=(ImaginaryUnit) { return false; }
   friend bool operator==(Real, ImaginaryUnit) { return false; }
   friend bool operator!=(Real, ImaginaryUnit) { return true; }
   friend bool operator==(ImaginaryUnit, Real) { return false; }
   friend bool operator!=(ImaginaryUnit, Real) { return true; }
};

extern ImaginaryUnit _I_;         // what is usually written as i or j


class CX
{
   Real X;
   Real Y;
public:
   CX(Real x1, Real y1) : X(x1), Y(y1) {}
   CX(Real x1) : X(x1), Y(0) {}
   CX() {}
   CX(const CX& z) : X(z.X), Y(z.Y) {}
   CX(ImaginaryUnit) : X(0.0), Y(1.0) {}
   CX(Imag y);
   CX(const Polar& p);

   void operator=(const CX& z1) { X = z1.X; Y = z1.Y; }
   void operator=(Real x1) { X = x1; Y = 0.0; }
   void operator=(ImaginaryUnit) { X = 0; Y = 1; }
   void operator=(Imag y);
   void operator=(const Polar& p);

   Real real() const { return X; }
   Real imag() const { return Y; }
   Real& real() { return X; }
   Real& imag() { return Y; }
   Imag imag_as_imag() const;
   CX conj() const { return CX(X,-Y); }
   CX operator+() const { return *this; }
   CX operator-() const { return CX(-X,-Y); }
   Real cabs() const;
   Real arg() const;
   Real sum_square() const { return X * X + Y * Y; }


   friend CX operator*(const CX& z1, const CX& z2);
   friend CX operator/(const CX& z1, const CX& z2);
   friend CX operator+(const CX& z1, const CX& z2);
   friend CX operator-(const CX& z1, const CX& z2);

   friend CX operator*(const CX& z1, Real x2);
   friend CX operator/(const CX& z1, Real x2);
   friend CX operator+(const CX& z1, Real x2);
   friend CX operator-(const CX& z1, Real x2);

   friend CX operator*(Real x1, const CX& z2);
   friend CX operator/(Real x1, const CX& z2);
   friend CX operator+(Real x1, const CX& z2);
   friend CX operator-(Real x1, const CX& z2);

   friend CX operator*(const CX& z1, ImaginaryUnit);
   friend CX operator/(const CX& z1, ImaginaryUnit);
   friend CX operator+(const CX& z1, ImaginaryUnit);
   friend CX operator-(const CX& z1, ImaginaryUnit);

   friend CX operator*(ImaginaryUnit, const CX& z2);
   friend CX operator/(ImaginaryUnit, const CX& z2);
   friend CX operator+(ImaginaryUnit, const CX& z2);
   friend CX operator-(ImaginaryUnit, const CX& z2);

   friend CX operator*(const CX& z1, Imag y2);
   friend CX operator/(const CX& z1, Imag y2);
   friend CX operator+(const CX& z1, Imag y2);
   friend CX operator-(const CX& z1, Imag y2);

   friend CX operator*(Imag y1, const CX& z2);
   friend CX operator/(Imag y1, const CX& z2);
   friend CX operator+(Imag y1, const CX& z2);
   friend CX operator-(Imag y1, const CX& z2);

   friend bool operator==(const CX& z1, const CX& z2);
   friend bool operator==(Real r1, const CX& z2);
   friend bool operator==(Imag y1, const CX& z2);
   friend bool operator==(ImaginaryUnit, const CX& z2);
   friend bool operator==(const CX& z1, Real r2);
   friend bool operator==(const CX& z1, Imag y2);
   friend bool operator==(const CX& z1, ImaginaryUnit);


   void operator*=(const CX& z1)
      { Real t = X; X = X*z1.X-Y*z1.Y; Y = t*z1.Y+Y*z1.X; }
   void operator/=(const CX& z1) { *this = *this/z1; }
   void operator+=(const CX& z1) { X += z1.X; Y += z1.Y; }
   void operator-=(const CX& z1) { X -= z1.X; Y -= z1.Y; }

   void operator*=(Real x1) { X *= x1; Y *= x1; }
   void operator/=(Real x1) { X /= x1; Y /= x1; }
   void operator+=(Real x1) { X += x1; }
   void operator-=(Real x1) { X -= x1; }

   void operator*=(ImaginaryUnit) { Real t = X; X = -Y; Y = t; }
   void operator/=(ImaginaryUnit) { Real t = X; X = Y; Y = -t; }
   void operator+=(ImaginaryUnit) { Y += 1; }
   void operator-=(ImaginaryUnit) { Y -= 1; }

   void operator*=(Imag y);
   void operator/=(Imag y);
   void operator+=(Imag y);
   void operator-=(Imag y);

   void operator*=(const Polar& p1);
   void operator/=(const Polar& p1);
   void operator+=(const Polar& p1);
   void operator-=(const Polar& p1);


   friend CX exp(const CX&);
   friend CX log(const CX&);
   friend Polar polar_exp(const CX&);

   friend Real norm1(const CX& z) { return ::fabs(z.X) + ::fabs(z.Y); }

   friend CX sqrt(const CX&);

   friend CX square(const CX& z)
      { return CX(z.X * z.X - z.Y * z.Y, 2 * z.X * z.Y); }

   friend CX pow(const CX& z1, int n2);
   friend CX pow(const CX& z1, Real r2);
   friend CX pow(const CX& z1, Imag y2);
   friend CX pow(const CX& z1, const CX& z2);
   friend CX pow(Real r1, const CX& z2);
   friend CX pow(Imag y1, const CX& z2);
   friend Polar pow(const Polar& p1, const CX& z2);

   friend CX sin(const CX&);
   friend CX cos(const CX&);
   friend CX tan(const CX&);
   friend CX sinh(const CX&);
   friend CX cosh(const CX&);
   friend CX tanh(const CX&);

   friend class Imag;
   friend class Polar;

};

class Imag
{
   Real Y;
//   Imag(Real z) : Y(z) {}
public:
   Imag(ConvertFromReal x) : Y(x.Value()) {}
   Imag(const Imag& z) : Y(z.Y) {}
   Imag() {}
   Imag(ImaginaryUnit) : Y(1.0) {}

   void operator=(const Imag& y1) { Y = y1.Y; }
   void operator=(ImaginaryUnit) { Y = 1.0; }

   Real real() const { return 0; }
   Real imag() const { return Y; }
   Real& imag() { return Y; }
   Imag imag_as_imag() const { return *this; }
   Imag conj() const { return Imag(-Y); }
   Imag operator+() const { return Imag(Y); }
   Imag operator-() const { return Imag(-Y); }
   Real cabs() const { return ::fabs(Y); }
   Real arg() const;
   Real sum_square() const { return Y * Y; }

   friend Imag operator*(Real x1, ImaginaryUnit);
   friend Imag operator/(Real x1, ImaginaryUnit);
   friend Imag operator*(ImaginaryUnit, Real x2);
   friend Imag operator/(ImaginaryUnit, Real x2);
   friend Imag operator+(ImaginaryUnit, ImaginaryUnit);

   friend Real operator*(Imag y1, Imag y2);
   friend Real operator/(Imag y1, Imag y2);
   friend Imag operator+(Imag y1, Imag y2);
   friend Imag operator-(Imag y1, Imag y2);

   friend Imag operator*(Imag y1, Real x2);
   friend Imag operator/(Imag y1, Real x2);
   friend CX operator+(Imag y1, Real x2);
   friend CX operator-(Imag y1, Real x2);

   friend Imag operator*(Real x1, Imag y2);
   friend Imag operator/(Real x1, Imag y2);
   friend CX operator+(Real x1, Imag y2);
   friend CX operator-(Real x1, Imag y2);

   friend CX operator*(const CX& z1, Imag y2);
   friend CX operator/(const CX& z1, Imag y2);
   friend CX operator+(const CX& z1, Imag y2);
   friend CX operator-(const CX& z1, Imag y2);

   friend CX operator*(Imag y1, const CX& z2);
   friend CX operator/(Imag y1, const CX& z2);
   friend CX operator+(Imag y1, const CX& z2);
   friend CX operator-(Imag y1, const CX& z2);

   friend Real operator*(Imag y1, ImaginaryUnit);
   friend Real operator/(Imag y1, ImaginaryUnit);
   friend Imag operator+(Imag y1, ImaginaryUnit);
   friend Imag operator-(Imag y1, ImaginaryUnit);

   friend Real operator*(ImaginaryUnit, Imag y2);
   friend Real operator/(ImaginaryUnit, Imag y2);
   friend Imag operator+(ImaginaryUnit, Imag y2);
   friend Imag operator-(ImaginaryUnit, Imag y2);

   friend Polar operator*(const Polar& p1, Imag y2);
   friend Polar operator/(const Polar& p1, Imag y2);
   friend Polar operator*(Imag y1, const Polar& p2);
   friend Polar operator/(Imag y1, const Polar& p2);

   friend bool operator==(Imag y1, Imag y2);
   friend bool operator==(Real r1, Imag y2);
   friend bool operator==(const CX& z1, Imag y2);
   friend bool operator==(const Polar& p1, Imag y2);
   friend bool operator==(ImaginaryUnit, Imag y2);
   friend bool operator==(Imag y1, const CX& z2);
   friend bool operator==(Imag y1, const Polar& p2);
   friend bool operator==(Imag y1, Real r2);
   friend bool operator==(Imag y1, ImaginaryUnit);



   void operator*=(Real x1) { Y *= x1; }
   void operator/=(Real x1) { Y /= x1; }
   void operator+=(Imag z) { Y += z.Y; }
   void operator-=(Imag z) { Y -= z.Y; }
   void operator+=(ImaginaryUnit) { Y += 1; }
   void operator-=(ImaginaryUnit) { Y -= 1; }

   friend CX exp(Imag);
   friend CX log(Imag);
   friend Polar polar_exp(Imag z);

   friend CX sqrt(Imag y);

   friend Real square(const Imag& y) { return - y.Y * y.Y; }

   friend CX pow(Imag y1, int n2);
   friend CX pow(Imag y1, Real r2);
   friend CX pow(Imag y1, Imag y2);
   friend CX pow(Real r1, Imag y2);


   friend Imag sin(Imag y) { return Imag(::sinh(y.Y)); }
   friend Real cos(Imag y) { return ::cosh(y.Y); }
   friend Imag tan(Imag y) { return Imag(::tanh(y.Y)); }
   friend Imag sinh(Imag y) { return Imag(::sin(y.Y)); }
   friend Real cosh(Imag y) { return ::cos(y.Y); }
   friend Imag tanh(Imag y) { return Imag(::tan(y.Y)); }

   friend class CX;
   friend class Polar;

};

inline ImaginaryUnit ImaginaryUnit::operator+() { return _I_; }
inline Imag ImaginaryUnit::operator-() { return Imag(-1); }

inline Real real(const CX& z) { return z.real(); }
inline Real imag(const CX& z) { return z.imag(); }
inline Imag imag_as_imag(const CX& z) { return z.imag_as_imag(); }
inline CX conj(const CX& z) { return z.conj(); }
inline Real cabs(const CX& z) { return z.cabs(); }
inline Real fabs(const CX& z) { return z.cabs(); }
inline Real arg(const CX& z) { return z.arg(); }
inline Real sum_square(const CX& z) { return z.sum_square(); }

inline Real real(Imag y) { return y.real(); }
inline Real imag(Imag y) { return y.imag(); }
inline Imag imag_as_imag(const Imag y) { return y; }
inline Imag conj(Imag y) { return y.conj(); }
inline Real cabs(Imag y) { return y.cabs(); }
inline Real fabs(Imag y) { return y.cabs(); }
inline Real arg(Imag y) { return y.arg(); }
inline Real sum_square(Imag y) { return y.sum_square(); }


// Real OP CX

inline CX operator*(Real x1, const CX& z2) { return CX(x1*z2.X, x1*z2.Y); }
//     CX operator/(Real x1, const CX& z2);
inline CX operator+(Real x1, const CX& z2) { return CX(x1+z2.X, z2.Y); }
inline CX operator-(Real x1, const CX& z2) { return CX(x1-z2.X, -z2.Y); }

// Real OP Imag

inline Imag operator*(Real x1, Imag y2) { return Imag(x1*y2.Y); }
inline Imag operator/(Real x1, Imag y2) { return Imag(-x1/y2.Y); }
inline CX operator+(Real x1, Imag y2) { return CX(x1, y2.Y); }
inline CX operator-(Real x1, Imag y2) { return CX(x1, -y2.Y); }

// Real OP ImaginaryUnit

inline Imag operator*(Real x1, const ImaginaryUnit) { return Imag(x1); }
inline Imag operator/(Real x1, const ImaginaryUnit) { return Imag(-x1); }
inline CX operator+(Real x1, const ImaginaryUnit) { return CX(x1, 1); }
inline CX operator-(Real x1, const ImaginaryUnit) { return CX(x1, -1); }

// CX OP Real

inline CX operator*(const CX& z1, Real x2) { return CX(z1.X*x2, z1.Y*x2); }
inline CX operator/(const CX& z1, Real x2) { return CX(z1.X/x2, z1.Y/x2); }
inline CX operator+(const CX& z1, Real x2) { return CX(z1.X+x2, z1.Y); }
inline CX operator-(const CX& z1, Real x2) { return CX(z1.X-x2, z1.Y); }

// CX OP CX

inline CX operator*(const CX& z1, const CX& z2)
   { return CX(z1.X*z2.X-z1.Y*z2.Y, z1.X*z2.Y+z1.Y*z2.X); }
//     CX operator/(const CX& z1, const CX& z2);
inline CX operator+(const CX& z1, const CX& z2)
   { return CX(z1.X+z2.X, z1.Y+z2.Y); }
inline CX operator-(const CX& z1, const CX& z2)
   { return CX(z1.X-z2.X, z1.Y-z2.Y); }

// CX OP Imag

inline CX operator*(const CX& z1, Imag y2)
   { return CX(-z1.Y*y2.Y, z1.X*y2.Y); }
inline CX operator/(const CX& z1, Imag y2)
   { return CX(z1.Y/y2.Y, -z1.X/y2.Y); }
inline CX operator+(const CX& z1, Imag y2) { return CX(z1.X, z1.Y+y2.Y); }
inline CX operator-(const CX& z1, Imag y2) { return CX(z1.X, z1.Y-y2.Y); }

// CX OP ImaginaryUnit

inline CX operator*(const CX& z1, ImaginaryUnit) { return CX(-z1.Y, z1.X); }
inline CX operator/(const CX& z1, ImaginaryUnit) { return CX(z1.Y, -z1.X); }
inline CX operator+(const CX& z1, ImaginaryUnit) { return CX(z1.X, z1.Y+1); }
inline CX operator-(const CX& z1, ImaginaryUnit) { return CX(z1.X, z1.Y-1); }

// Imag OP Real

inline Imag operator*(Imag y1, Real x2) { return Imag(y1.Y*x2); }
inline Imag operator/(Imag y1, Real x2) { return Imag(y1.Y/x2); }
inline CX operator+(Imag y1, Real x2) { return CX(x2, y1.Y); }
inline CX operator-(Imag y1, Real x2) { return CX(-x2, y1.Y); }

// Imag OP CX

inline CX operator*(Imag y1, const CX& z2)
   { return CX(-y1.Y*z2.Y, y1.Y*z2.X); }
//     CX operator/(Imag y1, const CX& z2);
inline CX operator+(Imag y1, const CX& z2) { return CX(z2.X, y1.Y+z2.Y); }
inline CX operator-(Imag y1, const CX& z2) { return CX(-z2.X, y1.Y-z2.Y); }

// Imag OP Imag

inline Real operator*(Imag y1, Imag y2) { return - y1.Y * y2.Y; }
inline Real operator/(Imag y1, Imag y2) { return y1.Y / y2.Y; }
inline Imag operator+(Imag y1, Imag y2) { return Imag(y1.Y+y2.Y); }
inline Imag operator-(Imag y1, Imag y2) { return Imag(y1.Y-y2.Y); }

// Imag OP ImaginaryUnit

inline Real operator*(Imag y1, ImaginaryUnit) { return - y1.Y; }
inline Real operator/(Imag y1, ImaginaryUnit) { return y1.Y; }
inline Imag operator+(Imag y1, ImaginaryUnit) { return Imag(y1.Y+1); }
inline Imag operator-(Imag y1, ImaginaryUnit) { return Imag(y1.Y-1); }

// ImaginaryUnit OP Real

inline Imag operator*(ImaginaryUnit, Real x2) { return Imag(x2); }
inline Imag operator/(ImaginaryUnit, Real x2) { return Imag(1.0/x2); }
inline CX operator+(ImaginaryUnit, Real x2) { return CX(x2, 1); }
inline CX operator-(ImaginaryUnit, Real x2) { return CX(-x2, 1); }

// ImaginaryUnit OP CX

inline CX operator*(ImaginaryUnit, const CX& z2) { return CX(-z2.Y, z2.X); }
//     CX operator/(ImaginaryUnit, const CX& z2);
inline CX operator+(ImaginaryUnit, const CX& z2) { return CX(z2.X, z2.Y+1); }
inline CX operator-(ImaginaryUnit, const CX& z2) { return CX(-z2.X, 1-z2.Y); }

// ImaginaryUnit OP Imag

inline Real operator*(ImaginaryUnit, Imag y2) { return - y2.Y; }
inline Real operator/(ImaginaryUnit, Imag y2) { return 1.0 / y2.Y; }
inline Imag operator+(ImaginaryUnit, Imag y2) { return Imag(1+y2.Y); }
inline Imag operator-(ImaginaryUnit, Imag y2) { return Imag(1-y2.Y); }

// ImaginaryUnit OP ImaginaryUnit

inline Real operator*(ImaginaryUnit, ImaginaryUnit) { return - 1; }
inline Real operator/(ImaginaryUnit, ImaginaryUnit) { return 1; }
inline Imag operator+(ImaginaryUnit, ImaginaryUnit) { return Imag(2); }
inline Real operator-(ImaginaryUnit, ImaginaryUnit) { return 0; }



inline CX::CX(Imag y) { X = 0; Y = y.Y; }
inline Imag CX::imag_as_imag() const { return Imag(Y); }
inline void CX::operator=(Imag y) { X = 0; Y = y.Y; }
inline void CX::operator*=(Imag y) { Real t = X; X = -Y*y.Y; Y = t*y.Y; }
inline void CX::operator/=(Imag y) { Real t = X; X = Y/y.Y; Y = -t/y.Y; }
inline void CX::operator+=(Imag y) { Y += y.Y; }
inline void CX::operator-=(Imag y) { Y -= y.Y; }

inline bool operator==(const CX& z1, const CX& z2)
   { return z1.X == z2.X && z1.Y == z2.Y; }
inline bool operator==(const CX& z1, Real r2)
   { return z1.X == r2 && z1.Y == 0.0; }
inline bool operator==(const CX& z1, Imag y2)
   { return z1.X == 0 && z1.Y == y2.Y; }
inline bool operator==(const CX& z1, ImaginaryUnit)
   { return z1.X == 0.0 && z1.Y == 1.0; }
inline bool operator==(Imag y1, const CX& z2)
   { return 0.0 == z2.X && y1.Y == z2.Y; }
inline bool operator==(Imag y1, Real r2) { return y1.Y == 0.0 && r2 == 0.0; }
inline bool operator==(Imag y1, Imag y2) { return y1.Y == y2.Y; }
inline bool operator==(Imag y1, ImaginaryUnit) { return y1.Y == 1.0; }
inline bool operator==(Real r1, const CX& z2)
   { return r1 == z2.X && 0.0 == z2.Y; }
inline bool operator==(Real r1, Imag y2) { return r1 == 0.0 && y2.Y == 0.0; }
inline bool operator==(ImaginaryUnit, const CX& z2)
   { return z2.X == 0.0 && z2.Y == 1.0; }
inline bool operator==(ImaginaryUnit, Imag y2) { return y2.Y == 1.0; }

inline bool operator!=(const CX& z1, const CX& z2) { return !(z1 == z2); }
inline bool operator!=(const CX& z1, Real r2) { return !(z1 == r2); }
inline bool operator!=(const CX& z1, Imag y2) { return !(z1 == y2); }
inline bool operator!=(const CX& z1, ImaginaryUnit) { return !(z1 == _I_); }
inline bool operator!=(Imag y1, const CX& z2) { return !(y1 == z2); }
inline bool operator!=(Imag y1, Real r2) { return !(y1 == r2); }
inline bool operator!=(Imag y1, Imag y2) { return !(y1 == y2); }
inline bool operator!=(Imag y1, ImaginaryUnit) { return !(y1 == _I_); }
inline bool operator!=(Real r1, const CX& z2) { return !(r1 == z2); }
inline bool operator!=(Real r1, Imag y2) { return !(r1 == y2); }
inline bool operator!=(ImaginaryUnit, const CX& z2) { return !(_I_ == z2); }
inline bool operator!=(ImaginaryUnit, Imag y2) { return !(_I_ == y2); }



// integers mod 4 representing angle category - see diagram in class polar

class Quadrant
{
   char q;
public:
   Quadrant() {}
   Quadrant(int i) : q((char)i) {}
   void operator=(int i) { q = (char)i; }
   void operator++() { q += (char)1; q &= (char)3; }
   void operator--() { q += (char)3; q &= (char)3; }
   Quadrant plus1() const { return (q + (char)1) & (char)3; }
   Quadrant plus2() const { return (q ^ (char)2); }
   Quadrant plus3() const { return (q + (char)3) & (char)3; }
   void pluseq2() { q ^= (char)2; }
   Quadrant h_reflect_minus1() const { return q ^ (char)3; }
   Quadrant h_reflect() const { return ((char)4 - q) & (char)3; }
   Quadrant h_reflect_plus1() const { return q ^ (char)1; }
   Quadrant v_reflect() const { return ((char)6 - q) & (char)3; }
   operator int() const { return (int)q; }
   void operator+=(Quadrant q2) { q += q2.q; q &= (char)3; }
   void operator-=(Quadrant q2) { q += (char)4; q -= q2.q; q &= (char)3; }
   friend Quadrant operator+(Quadrant q1, Quadrant q2)
      { return (q1.q + q2.q) & (char)3; }
   friend Quadrant operator-(Quadrant q1, Quadrant q2)
      { return (4 + q1.q - q2.q) & (char)3; }
   friend Quadrant operator+(Quadrant q1, int i)
      { return (q1.q + (char)i) & (char)3; }
   friend Quadrant operator-(Quadrant q1, int i)
      { return (4 + q1.q - (char)i) & (char)3; }
};

class Polar
{
   Quadrant RA;            // number of right angles, 0, 1, 2, 3;
   Real R, Theta;          // our theta will be between -pi/4 and pi/4;
                           // add pi/2 * RA and normalise to get angle

   //                      RA = 1
   //                  .     |     .
   //                    .   |   .
   //                      . | .
   //    RA = 2  ------------.------------   RA = 0
   //                      . | .
   //                    .   |   .
   //                  .     |     .
   //                      RA = 3

   Polar(Quadrant ra, Real r, Real theta) : RA(ra), R(r), Theta(theta) {}

public:
   Polar() {}
   Polar(Real x);
   Polar(Real r, Real theta);
   Polar(const CX& z);
   Polar(const Polar& p) : RA(p.RA), R(p.R), Theta(p.Theta) {}
   Polar(Imag y);
   Polar(ImaginaryUnit) : RA(1), R(1.0), Theta(0.0) {}

   void operator=(const Polar& p) { RA = p.RA; R = p.R; Theta = p.Theta; }
   void operator=(const CX& z) { *this = Polar(z); }
   void operator=(Real x1);
   void operator=(ImaginaryUnit) { RA = 1; R = 1.0; Theta = 0.0; }
   void operator=(Imag y);

   Real theta() const { return Theta; }
   Real& theta() { return Theta; }
   Quadrant quadrant() const { return RA; }
   Quadrant& quadrant() { return RA; }
   Real cabs() const { return R; }
   Real& cabs() { return R; }
   Real sum_square() const { return R * R; }

   Real real() const;
   Real imag() const;
   Imag imag_as_imag() const { return Imag(imag()); }
   Polar conj() const { return Polar(RA.h_reflect(), R, -Theta); }
   Real arg() const;
   Polar operator+() const { return *this; }
   Polar operator-() const { return Polar(RA.plus2(), R, Theta); }
   void assert_is_valid() const;
   void AssertIsValid() const { assert_is_valid(); }

   friend Polar operator*(const Polar& p1, const Polar& p2);
   friend Polar operator/(const Polar& p1, const Polar& p2);
   friend CX operator+(const Polar& p1, const Polar& p2);
   friend CX operator-(const Polar& p1, const Polar& p2);

   friend CX operator+(const Polar& p1, const CX& z2);
   friend CX operator-(const Polar& p1, const CX& z2);
   friend CX operator+(const CX& z1, const Polar& p2);
   friend CX operator-(const CX& z1, const Polar& p2);


   friend Polar operator*(const Polar& p1, Real x2);
   friend Polar operator/(const Polar& p1, Real x2);
   friend CX operator+(const Polar& p1, Real x2);
   friend CX operator-(const Polar& p1, Real x2);
   friend Polar operator*(Real x1, const Polar& p2) { return p2 * x1; }
   friend Polar operator/(Real x1, const Polar& p2);
   friend CX operator+(Real x1, const Polar& p2);
   friend CX operator-(Real x1, const Polar& p2);

   friend Polar operator*(const Polar& p1, Imag y2);
   friend Polar operator/(const Polar& p1, Imag y2);
   friend CX operator+(const Polar& p1, Imag y2);
   friend CX operator-(const Polar& p1, Imag y2);
   friend Polar operator*(Imag y1, const Polar& p2) { return p2 * y1; }
   friend Polar operator/(Imag y1, const Polar& p2);
   friend CX operator+(Imag y1, const Polar& p2);
   friend CX operator-(Imag y1, const Polar& p2);

   friend Polar operator*(const Polar& p1, ImaginaryUnit)
      { return Polar(p1.RA.plus1(), p1.R, p1.Theta); }
   friend Polar operator/(const Polar& p1, ImaginaryUnit)
      { return Polar(p1.RA.plus3(), p1.R, p1.Theta); }
   friend CX operator+(const Polar& p1, ImaginaryUnit);
   friend CX operator-(const Polar& p1, ImaginaryUnit);
   friend Polar operator*(ImaginaryUnit, const Polar& p2)
      { return Polar(p2.RA.plus1(), p2.R, p2.Theta); }
   friend Polar operator/(ImaginaryUnit, const Polar& p2)
      { return Polar(p2.RA.h_reflect_plus1(), 1.0 / p2.R, -p2.Theta); }
   friend CX operator+(ImaginaryUnit, const Polar& p2);
   friend CX operator-(ImaginaryUnit, const Polar& p2);

   friend bool operator==(const Polar& p1, const Polar& p2);
   friend bool operator==(Real r1, const Polar& p2);
   friend bool operator==(Imag y1, const Polar& p2);
   friend bool operator==(ImaginaryUnit, const Polar& p2);
   friend bool operator==(const Polar& p1, Real r2);
   friend bool operator==(const Polar& p1, Imag y2);
   friend bool operator==(const Polar& p1, ImaginaryUnit);




   void operator*=(const Polar& p);
   void operator/=(const Polar& p);
   void operator+=(const Polar& p);
   void operator-=(const Polar& p);

   void operator*=(const CX& p);
   void operator/=(const CX& p);
   void operator+=(const CX& p);
   void operator-=(const CX& p);

   void operator*=(Real x);
   void operator/=(Real x);
   void operator+=(Real x);
   void operator-=(Real x);

   void operator*=(Imag y);
   void operator/=(Imag y);
   void operator+=(Imag y);
   void operator-=(Imag y);

   void operator*=(ImaginaryUnit) { ++RA; }
   void operator/=(ImaginaryUnit) { --RA; }
   void operator+=(ImaginaryUnit);
   void operator-=(ImaginaryUnit);

   friend CX log(const Polar&);

   friend Polar sqrt(const Polar& p);

   friend Polar square(const Polar& p) { return p * p; }

   friend Polar pow(const Polar& p1, int n2);
   friend Polar pow(const Polar& p1, Real r2);
   friend Polar pow(const Polar& p1, Imag y2);
   friend Polar pow(const Polar& p1, const CX& z2);

   friend class CX;
   friend class Imag;

};

inline Real real(const Polar& p) { return p.real(); }
inline Real imag(const Polar& p) { return p.imag(); }
inline Imag imag_as_imag(const Polar& p) { return p.imag_as_imag(); }
inline Polar conj(const Polar& p) { return p.conj(); }
inline Real cabs(const Polar& p) { return p.cabs(); }
inline Real fabs(const Polar& p) { return p.cabs(); }
inline Real arg(const Polar& p) { return p.arg(); }
inline Real sum_square(const Polar& p) { return p.sum_square(); }


inline Polar polar_exp(Imag z) { return Polar(1.0, z.Y); }

inline bool operator==(ImaginaryUnit, const Polar& p2)
   { return p2.RA == 1 && p2.R == 1.0 && p2.Theta == 0.0; }
inline bool operator==(const Polar& p1, ImaginaryUnit)
   { return p1.RA == 1 && p1.R == 1.0 && p1.Theta == 0.0; }

inline bool operator==(Real r1, const Polar& p2) { return p2 == r1; }
inline bool operator==(Imag y1, const Polar& p2) { return p2 == y1; }


inline bool operator!=(const Polar& p1, const Polar& p2) { return !(p1 == p2); }
inline bool operator!=(Real r1, const Polar& p2) { return !(p2 == r1); }
inline bool operator!=(Imag y1, const Polar& p2) { return !(p2 == y1); }
inline bool operator!=(ImaginaryUnit, const Polar& p2) { return !(p2 == _I_); }
inline bool operator!=(const Polar& p1, Real r2) { return !(p1 == r2); }
inline bool operator!=(const Polar& p1, Imag y2) { return !(p1 == y2); }
inline bool operator!=(const Polar& p1, ImaginaryUnit) { return !(p1 == _I_); }


//make sure constants are available to any global function

class ComplexPackageInitialise
{
public:
   ComplexPackageInitialise();
};

static ComplexPackageInitialise DoComplexPackageInitialise;

#ifdef use_namespace
}
#endif


#endif

// body file: cx.cpp
// body file: cx_polar.cpp



