// file array1.h

// A very simple templated array class
// The C language doesn't provide a satisfactory dynamic array facility;
// This is a very simple C++ one
// do 1 and 2 dimensions

// cut down version

#include "include.h"
#include "myexcept.h"

#ifndef ARRAY1_H                        // don't include twice
#define ARRAY1_H

#ifdef use_namespace
using namespace RBD_COMMON;
#endif




template <class T>
class array1
{
private:
protected:
   T* a;                                // pointer to the array
   int n;                               // length of the array
public:
   array1() : a(0), n(0) {}             // build an array, length zero
   array1(int xn);                      // build an array length xn
   ~array1();                           // return the space to memory
   T& operator()(int i);                // access element of the array
   T operator()(int i) const;           // access element of constant array
   void operator=(T ai);                // set the array equal to a constant
   void operator=(const array1<T>& b);  // copy the elements of an array
   array1(const array1<T>& b);          // new array equal to an existing one
   int size() const { return n; }       // return the size of the array
   void resize(int xn);                 // change the size of the array
   T* data() { return a; }              // pointer to the data
   const T* data() const { return a; }  // pointer to the data
};

// 2 dimensional array class

template <class T>
class array2 : public array1<T>
{
protected:
   int n_rows, n_cols;
public:
   array2(int x_n_rows, int x_n_cols);
   T& operator()(int i, int j);
			      // access element of the array
   T operator()(int i, int j) const;
			      // access element of constant array
   void operator=(T ai);    // set the array equal to a constant
   void operator=(const array2<T>& b);
			      // copy the elements of an array
   array2(const array2<T>& b);
			      // make a new array equal to an existing one
   int N_rows() const { return n_rows; }
   int N_cols() const { return n_cols; }
};




// construct a new array of length xn. Check that xn is non-negative and
// that space is available

template <class T>
array1<T>::array1(int xn) : n(xn)
{
   if (n < 0) Throw(Logic_error("invalid array length"));
   else if (n == 0) a = 0;
   else
   {
      a = new T [n];
      if (!a) Throw(Bad_alloc());
   }
}

// destroy an array - return its space to memory

template <class T>
array1<T>::~array1()
   { if (a) delete [] a; }

// access an element of an array; return a "reference" so elements
// can be modified.
// check index is within range
// in this array class the index runs from 0 to n-1

template <class T>
void array1<T>::resize(int xn)
{
   if (a) delete [] a;
   n = xn;
   if (n < 0) Throw(Logic_error("invalid array length"));
   else if (n == 0) a = 0;
   else
   {
      a = new T [n];
      if (!a) Throw(Bad_alloc());
   }
}

template <class T>
T& array1<T>::operator()(int i)
{
   if (i < 0 || i >= n) Throw(Logic_error("array index out of range"));
   return a[i];
}

// same thing again but for arrays declared constant so we can't
// modify its elements

template <class T>
T array1<T>::operator()(int i) const
{
   if (i < 0 || i >= n) Throw(Logic_error("array index out of range"));
   return a[i];
}

// set all the elements equal to a given value

template <class T>
void array1<T>::operator=(T ai)
   {  for (int i = 0; i < n; i++) a[i] = ai; }

// set the elements equal to those of another array.
// check the arrays are of the same length

template <class T>
void array1<T>::operator=(const array1<T>& b)
{
   if (b.n != n) Throw(Logic_error("array lengths differ in copy"));
   for (int i = 0; i < n; i++) a[i] = b.a[i];
}

// construct a new array equal to an existing array
// check that space is available

template <class T>
array1<T>::array1(const array1<T>& b) : n(b.n)
{
   if (n == 0) a = 0;
   else
   {
      a = new T [n];
      if (!a) Throw(Bad_alloc());
      for (int i = 0; i < n; i++) a[i] = b.a[i];
   }
}



// two dimensional array

template <class T>
array2<T>::array2(int x_n_rows, int x_n_cols)
   : array1<T>(x_n_rows*x_n_cols), n_rows(x_n_rows), n_cols(x_n_cols) {}

template <class T>
T& array2<T>::operator()(int i, int j)
{
   if (i < 0 || i >= n_rows || j < 0 || j >= n_cols)
      Throw(Logic_error("array index out of range"));
   return array1<T>::a[i+j*n_rows];
}

template <class T>
T array2<T>::operator()(int i, int j) const
{
   if (i < 0 || i >= n_rows || j < 0 || j >= n_cols)
      Throw(Logic_error("array index out of range"));
   return array1<T>::a[i+j*n_rows];
}

template <class T>
void array2<T>::operator=(T ai) { array1<T>::operator=(ai); }

template <class T>
void array2<T>::operator=(const array2<T>& b)
{
   if (n_rows != b.n_rows && n_cols != b.n_cols)
      Throw(Logic_error("array lengths differ in copy"));
      for (int i = 0; i < array1<T>::n; i++) array1<T>::a[i] = b.a[i];
}

template <class T>
array2<T>::array2(const array2<T>& b)
   : array1<T>(b), n_rows(b.n_rows), n_cols(b.n_cols) {}





#endif

